/***********************************************************

 Copyright Derrick Stolee 2012.

 This file is part of SearchLib.

 SearchLib is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 SearchLib is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with SearchLib.  If not, see <http://www.gnu.org/licenses/>.

 *************************************************************/

/*
 * MMSImplementationGLPK.cpp
 *
 *  Created on: Jul 24, 2012
 *      Author: stolee
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stack>
#include "Set.hpp"
#include "translation.hpp"
#include "MMSLightweightManager.hpp"

#include "glpk.h"

void MMSLightweightManager::initLP()
{
	/*  initialize the LP */
	glp_term_hook(NULL, NULL);
	glp_term_out(0);

	this->lp = glp_create_prob();
	glp_add_cols(this->lp, this->n);

	this->parm = (glp_smcp*) malloc(sizeof(glp_smcp));
	glp_init_smcp(this->parm);
	this->parm->msg_lev = GLP_MSG_OFF; //GLP_MSG_ERR
	this->parm->meth = GLP_DUALP;
	this->parm->presolve = GLP_ON;

	int* row_indices = (int*) malloc((this->n + 1) * sizeof(int));
	double* row_values = (double*) malloc((this->n + 1) * sizeof(double));
	for ( int i = 1; i <= this->n; i++ )
	{
		// every variable is unbounded
		glp_set_col_bnds(this->lp, i, GLP_FR, -100.0, 100.0);

		row_indices[i] = i;
		row_values[i] = 1.0;
	}

	/* make the sum have value 0 */
	glp_add_rows(this->lp, 1);
	glp_set_row_name(lp, 1, "sum");
	glp_set_row_bnds(lp, 1, GLP_LO, 0.0, 0.0);
	glp_set_mat_row(this->lp, 1, this->n, row_indices, row_values);

	glp_add_rows(this->lp, this->n - 1);
	for ( int i = 1; i < this->n; i++ )
	{
		char buffer[500];
		int row_is[3];
		double row_vals[3];

		/* x_i - x_{i+1} >= 0 */
		row_is[0] = 0;
		row_is[1] = i;
		row_is[2] = i + 1;

		row_vals[0] = 0.0;
		row_vals[1] = 1.0;
		row_vals[2] = -1.0;

		sprintf(buffer, "ineq%d", i);

		glp_set_row_name(lp, 1 + i, buffer);
		glp_set_mat_row(this->lp, 1 + i, 2, row_is, row_vals);
		glp_set_row_bnds(lp, 1 + i, GLP_LO, 0.0, 0.0);
	}

	this->num_rows = this->n;

	this->start_propagation_constraints = this->n;
	this->start_temporary_constraints = this->n;

	free(row_indices);
	free(row_values);

	// objective function
	glp_set_obj_dir(this->lp, GLP_MIN);
	glp_set_obj_coef(this->lp, 1, 1.0);

	this->solution = (double*) malloc(this->n * sizeof(double));

}

void MMSLightweightManager::freeLP()
{
	/* free the LP */
	if ( this->lp != 0 )
	{
		glp_delete_prob(this->lp);
		free(this->solution);

		free(this->parm);
	}
}

void MMSLightweightManager::clearLP()
{
	/* clear the LP */

	int* row_indices = (int*) malloc((this->num_rows + 1) * sizeof(int));

	for ( int i = 1; i <= this->num_rows - this->n; i++ )
	{
		row_indices[i] = i + this->n;
	}

	if ( this->num_rows - this->n > 0 )
	{
		glp_del_rows(this->lp, this->num_rows - this->n, row_indices);
	}

	this->num_rows = this->n;
	this->start_propagation_constraints = this->n;
	this->start_temporary_constraints = this->n;

	free(row_indices);
}

void MMSLightweightManager::rebuildLP()
{
	clock_t start_clock = clock();
	// start from scratch
	this->clearLP();

	// add positive constraints
	int* set = (int*) malloc(this->k * sizeof(int));
	int* neighbors = (int*) malloc(this->k * sizeof(int));
	for ( int i = 0; i < this->num_branchpos; i++ )
	{
		int set_rank = this->branchpos[i];
		INDEX_TO_SET(this->n, this->k, set_rank, set);
		int right_degree = this->getRightNeighborRanks(set, neighbors);

		bool is_minimal_positive = true;
		for ( int j = 0; is_minimal_positive && j < right_degree; j++ )
		{
			if ( this->set_labels[neighbors[j]] == LABEL_C_POS )
			{
				is_minimal_positive = false;
			}
		}

		if ( is_minimal_positive )
		{
			this->addPositiveBranchConstraint(set);
		}
	}
	for ( int i = 0; i < this->num_branchneg; i++ )
	{
		int set_rank = this->branchneg[i];
		INDEX_TO_SET(this->n, this->k, set_rank, set);
		int left_degree = this->getLeftNeighborRanks(set, neighbors);

		bool is_maximal_negative = true;
		for ( int j = 0; is_maximal_negative && j < left_degree; j++ )
		{
			if ( this->set_labels[neighbors[j]] == LABEL_C_NEG )
			{
				is_maximal_negative = false;
			}
		}

		if ( is_maximal_negative )
		{
			this->addNegativeBranchConstraint(set);
		}
	}

	for ( int i = 0; i < this->num_genneg; i++ )
	{
		int set_rank = this->genneg[i];
		INDEX_TO_SET(this->n, this->k, set_rank, set);
		int left_degree = this->getLeftNeighborRanks(set, neighbors);

		bool is_maximal_negative = true;
		for ( int j = 0; is_maximal_negative && j < left_degree; j++ )
		{
			if ( this->set_labels[neighbors[j]] == LABEL_C_NEG )
			{
				is_maximal_negative = false;
			}
		}

		if ( is_maximal_negative )
		{
			this->addNegativeBranchConstraint(set);
		}
	}

	free(set);
	free(neighbors);

	this->seconds_in_lp_reset =
	        this->seconds_in_lp_reset + ((double) (clock() - start_clock)) / (double) CLOCKS_PER_SEC;
}

void MMSLightweightManager::removeTemporaryConstraints()
{
	int rows_to_delete = this->num_rows - this->start_temporary_constraints;

	if ( rows_to_delete > 0 )
	{
		int* row_indices = (int*) malloc((this->num_rows + 1) * sizeof(int));

		for ( int i = 0; i < rows_to_delete; i++ )
		{
			/* 1 indexing is weird... */
			row_indices[i + 1] = this->start_temporary_constraints + i;
		}

		glp_del_rows(this->lp, rows_to_delete, row_indices);
		free(row_indices);
		this->num_rows = this->num_rows - rows_to_delete;
	}

}

void MMSLightweightManager::clearPropagationConstraints()
{
	this->removeTemporaryConstraints();

	int rows_to_delete = this->num_rows - this->start_propagation_constraints;

	if ( rows_to_delete > 0 )
	{
		int* row_indices = (int*) malloc((this->num_rows + 1) * sizeof(int));

		for ( int i = 0; i < rows_to_delete; i++ )
		{
			/* 1 indexing is weird... */
			row_indices[i + 1] = this->start_propagation_constraints + i;
		}

		glp_del_rows(this->lp, rows_to_delete, row_indices);
		free(row_indices);
		this->num_rows = this->num_rows - rows_to_delete;
	}

	this->start_temporary_constraints = this->start_propagation_constraints;
}

void MMSLightweightManager::removeTopBranchConstraint()
{
	this->clearPropagationConstraints();

	int rows_to_delete = 1;

	int* row_indices = (int*) malloc((2) * sizeof(int));

	/* 1 indexing is weird... */
	row_indices[1] = this->num_rows;

	glp_del_rows(this->lp, rows_to_delete, row_indices);
	free(row_indices);
	this->num_rows = this->num_rows - 1;

	this->start_propagation_constraints = this->num_rows;
	this->start_temporary_constraints = this->start_propagation_constraints;
}

void MMSLightweightManager::addPositiveBranchConstraint( int* set )
{
	this->clearPropagationConstraints();

	/* Add constraints to the LP */
	int* row_indices = (int*) malloc((this->k + 1) * sizeof(int));
	double* row_values = (double*) malloc((this->k + 1) * sizeof(double));

	row_indices[0] = 0;
	row_values[0] = 0.0;

	/* need to encode sum_{i \in S} x_i \geq 0 */
//	printf("-- adding POSITIVE CONSTRAINT: ");
	for ( int i = 0; i < this->k; i++ )
	{
//		printf("%2d ", set[i] + 1);
		row_indices[i + 1] = set[i] + 1;
		row_values[i + 1] = 1.0;
	}
//	printf("\n");

	this->num_rows = this->num_rows + 1;
	this->start_propagation_constraints = this->num_rows;
	this->start_temporary_constraints = this->num_rows;

	glp_add_rows(this->lp, 1);
	glp_set_mat_row(this->lp, this->num_rows, this->k, row_indices, row_values);
	glp_set_row_bnds(this->lp, this->num_rows, GLP_LO, 0.0, 0.0);

	free(row_indices);
	free(row_values);
}

void MMSLightweightManager::addNegativeBranchConstraint( int* set )
{
	// make sure these are clear
	this->clearPropagationConstraints();

	/* Add constraints to the LP */
	int* row_indices = (int*) malloc((this->k + 1) * sizeof(int));
	double* row_values = (double*) malloc((this->k + 1) * sizeof(double));

	row_indices[0] = 0;
	row_values[0] = 0.0;

	/* need to encode sum_{i \in S} x_i \leq -1 */
//	printf("-- adding NEGATIVE BRANCH CONSTRAINT: ");
	for ( int i = 0; i < this->k; i++ )
	{
//		printf("%2d ", set[i] + 1);
		row_indices[i + 1] = set[i] + 1;
		row_values[i + 1] = 1.0;
	}
//	printf("\n");

	this->num_rows = this->num_rows + 1;
	this->start_propagation_constraints = this->num_rows;
	this->start_temporary_constraints = this->num_rows;

	glp_add_rows(this->lp, 1);
	glp_set_mat_row(this->lp, this->num_rows, this->k, row_indices, row_values);
	glp_set_row_bnds(this->lp, this->num_rows, GLP_UP, -1.0, -1.0);

	free(row_indices);
	free(row_values);
}

void MMSLightweightManager::addNegativePropagationConstraint( int* set )
{
	// make sure these are clear
	this->removeTemporaryConstraints();

	/* Add constraints to the LP */
	int* row_indices = (int*) malloc((this->k + 1) * sizeof(int));
	double* row_values = (double*) malloc((this->k + 1) * sizeof(double));

	row_indices[0] = 0;
	row_values[0] = 0.0;

	/* need to encode sum_{i \in S} x_i \leq -1 */
//	printf("-- adding NEGATIVE PROPAGATION CONSTRAINT: ");
	for ( int i = 0; i < this->k; i++ )
	{
//		printf("%2d ", set[i] + 1);
		row_indices[i + 1] = set[i] + 1;
		row_values[i + 1] = 1.0;
	}
//	printf("\n");

	this->num_rows = this->num_rows + 1;
	this->start_temporary_constraints = this->num_rows;

	glp_add_rows(this->lp, 1);
	glp_set_mat_row(this->lp, this->num_rows, this->k, row_indices, row_values);
	glp_set_row_bnds(this->lp, this->num_rows, GLP_UP, -1.0, -1.0);

	free(row_indices);
	free(row_values);
}

void MMSLightweightManager::addNegativeTemporaryConstraint( int* set )
{
	/* Add constraints to the LP */
	int* row_indices = (int*) malloc((this->k + 1) * sizeof(int));
	double* row_values = (double*) malloc((this->k + 1) * sizeof(double));

	row_indices[0] = 0;
	row_values[0] = 0.0;

	/* need to encode sum_{i \in S} x_i \leq -1 */
//	printf("-- adding NEGATIVE TEMPORARY CONSTRAINT: ");
	for ( int i = 0; i < this->k; i++ )
	{
//		printf("%2d ", set[i] + 1);
		row_indices[i + 1] = set[i] + 1;
		row_values[i + 1] = 1.0;
	}
//	printf("\n");

	this->num_rows = this->num_rows + 1;

	glp_add_rows(this->lp, 1);
	glp_set_mat_row(this->lp, this->num_rows, this->k, row_indices, row_values);
	glp_set_row_bnds(this->lp, this->num_rows, GLP_UP, -1.0, -1.0);

	free(row_indices);
	free(row_values);
}

bool MMSLightweightManager::isLPFeasible()
{
	this->rebuildLP();

	glp_adv_basis(this->lp, 0);
	int result = glp_exact(this->lp, this->parm);

	if ( result != 0 )
	{
		printf("--The Solver had an error! %d\n", result);
	}

	result = glp_get_status(this->lp);
	if ( result == GLP_INFEAS || result == GLP_NOFEAS )
	{
		return false;
	}

	return true;
}

bool MMSLightweightManager::isLPFeasibleWithConstraint( int* set, int direction )
{
	this->rebuildLP();

	if ( direction < 0 )
	{
		this->addNegativeBranchConstraint(set);
	}
	else
	{
		this->addPositiveBranchConstraint(set);
	}

	glp_adv_basis(this->lp, 0);
	int result = glp_exact(this->lp, this->parm);

	if ( result != 0 )
	{
		printf("--The Solver had an error! %d\n", result);
	}

	result = glp_get_status(this->lp);
	if ( result == GLP_INFEAS || result == GLP_NOFEAS )
	{
		return false;
	}

	return true;
}

bool MMSLightweightManager::isLPOptimalACounterexample()
{
	//  check for counterexample!
	for ( int i = 0; i < this->n; i++ )
	{
		this->solution[i] = glp_get_col_prim(this->lp, i + 1);
	}

	/* check all sets */
	int* set = (int*) malloc(this->k * sizeof(int));

	int num_definitely_pos = 0;
	int num_pos = 0;

	INDEX_TO_SET(this->n, this->k, 0, set);
	for ( int rank = 0; rank < this->nchoosek; rank++ )
	{
		double sum = 0.0;

		for ( int j = 0; j < this->k; j++ )
		{
			sum += solution[set[j]];
		}

		if ( sum >= 0.0 )
		{
			num_definitely_pos++;
		}
		if ( sum >= -0.000001 )
		{
			num_pos++;
		}

		GET_SUCCESSOR(this->n, this->k, set);
	}

	free(set);

	this->sol_pos_sets = num_pos;

	if ( num_definitely_pos >= this->mms_target )
	{
		return false;
	}

	if ( num_pos >= this->mms_target )
	{
		printf("-- THE FOLLOWING OPTIMAL VECTOR HAS %d NON-NEGATIVE SETS, and %d SETS >= 0.00001 (Target: %d):\n",
		       num_definitely_pos, num_pos, this->mms_target);
		this->printLPOptimal();
		return false;
	}

	return true;
}

void MMSLightweightManager::snapshotLP()
{
	//defunct
}
void MMSLightweightManager::rollbackLP()
{
	//defunct
}

void MMSLightweightManager::printLPOptimal()
{
	for ( int i = 0; i < this->n; i++ )
	{
		this->solution[i] = glp_get_col_prim(this->lp, i + 1);
	}

	printf("optimal: \n");
	for ( int i = 0; i < this->n; i++ )
	{
		printf("\tx_{%2d} = %3.6lf\n", i, this->solution[i]);
	}
	printf(" with %d non-negative sets (target: %d)\n\n", this->sol_pos_sets, this->mms_target);
}

