/*
 * MMSLightweightManager.cpp
 *
 *  Created on: Jul 23, 2012
 *      Author: stolee
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stack>
#include <queue>
#include "Set.hpp"
#include "translation.hpp"
#include "shiftcalculations.hpp"
#include "MMSLightweightManager.hpp"

MMSLWNode::MMSLWNode( LONG_T label ) :
		SearchNode(label)
{
	this->branch_rank = -1;
	this->tested_stochastic = false;
}

MMSLWNode::~MMSLWNode()
{

}

/**
 * The left- and right- neighbors define the Hasse diagram of
 * the left-shift poset.
 * Given as _ranks_, not sets.
 * Index is always k*rank, even if not all previous ranks had k neighbors.
 */
int MMSLightweightManager::getRightNeighborRanks( int* set, int* ranks )
{
	int num_neighbors = 0;
	int* rset = (int*) malloc(this->k * sizeof(int));
	int rrank = 0;

	bcopy(set, rset, this->k * sizeof(int));

	for ( int j = 0; j < this->k; j++ )
	{
		/* can we increase this value? */
		bool is_neighbor = false;
		if ( j == this->k - 1 )
		{
			if ( set[j] < this->n - 1 )
			{
				is_neighbor = true;
			}
		}
		else
		{
			/* j  < k-1 */
			/* will increasing this value push us too far? */
			if ( set[j] + 1 < set[j + 1] )
			{
				is_neighbor = true;
			}
		}

		if ( is_neighbor )
		{
			rset[j] = set[j] + 1;
			ranks[num_neighbors] = INDEX_OF_SET(this->n, this->k, rset);
			num_neighbors++;

			/* reset */
			rset[j] = set[j];
		}
	}
	free(rset);

	return num_neighbors;
}

int MMSLightweightManager::getLeftNeighborRanks( int* set, int* ranks )
{
	int num_neighbors = 0;
	int* rset = (int*) malloc(this->k * sizeof(int));
	int rrank = 0;

	bcopy(set, rset, this->k * sizeof(int));

	for ( int j = 0; j < this->k; j++ )
	{
		/* can we decrease this value? */
		bool is_neighbor = false;
		if ( j == 0 )
		{
			if ( set[j] >= 1 )
			{
				is_neighbor = true;
			}
		}
		else
		{
			/* j  > 0 */
			/* will decreasing this value push us too far? */
			if ( set[j] - 1 > set[j - 1] )
			{
				is_neighbor = true;
			}
		}

		if ( is_neighbor )
		{
			rset[j] = set[j] - 1;
			ranks[num_neighbors] = INDEX_OF_SET(this->n, this->k, rset);
			num_neighbors++;
			/* reset */
			rset[j] = set[j];
		}
	}
	free(rset);

	return num_neighbors;
}

int MMSLightweightManager::getMeetRank( int num_sets, int* set_ranks )
{
	int* meetset = (int*) malloc(this->k * sizeof(int));
	int* tempset = (int*) malloc(this->k * sizeof(int));
	for ( int j = 0; j < this->k; j++ )
	{
		meetset[j] = 0;
	}

	for ( int i = 0; i < num_sets; i++ )
	{
		INDEX_TO_SET(this->n, this->k, set_ranks[i], tempset);

		for ( int j = 0; j < this->k; j++ )
		{
			if ( tempset[j] > meetset[j] )
			{
				meetset[j] = tempset[j];
			}
		}
	}

	int meetrank = INDEX_OF_SET(this->n, this->k, meetset);
	free(meetset);
	free(tempset);

	return meetrank;
}

int MMSLightweightManager::getJoinRank( int num_sets, int* set_ranks )
{
	int* joinset = (int*) malloc(this->k * sizeof(int));
	int* tempset = (int*) malloc(this->k * sizeof(int));
	for ( int j = 0; j < this->k; j++ )
	{
		joinset[j] = this->n;
	}

	for ( int i = 0; i < num_sets; i++ )
	{
		INDEX_TO_SET(this->n, this->k, set_ranks[i], tempset);
		for ( int j = 0; j < this->k; j++ )
		{
			if ( tempset[j] < joinset[j] )
			{
				joinset[j] = tempset[j];
			}
		}
	}

	int joinrank = INDEX_OF_SET(this->n, this->k, joinset);
	free(joinset);
	free(tempset);

	return joinrank;
}

int MMSLightweightManager::initLeftShift()
{
	this->left_shift = (int*) malloc(this->nchoosek * sizeof(int));
	bzero(this->left_shift, this->nchoosek * sizeof(int));

	fill_leftshift_values_form_colex(this->n, this->k, this->left_shift);

	/* the values will be calculated as required */
	return this->nchoosek * sizeof(int);
}

void MMSLightweightManager::freeLeftShift()
{
	if ( this->left_shift != 0 )
	{
		free(this->left_shift);
		this->left_shift = 0;
	}
}

int MMSLightweightManager::initCStarValues()
{
	this->left_c_star = (int*) malloc(this->nchoosek * sizeof(int));

	for ( int i = 0; i < this->nchoosek; i++ )
	{
		this->left_c_star[i] = this->left_shift[i];
	}

	return this->nchoosek * sizeof(int);
}

void MMSLightweightManager::freeCStarValues()
{
	if ( this->left_c_star != 0 )
	{
		free(this->left_c_star);
		this->left_c_star = 0;
	}
}

void MMSLightweightManager::clearCStarValues()
{
	/* reset the values to initial */
	for ( int i = 0; i < this->nchoosek; i++ )
	{
		this->left_c_star[i] = this->left_shift[i];
	}

	this->num_branchpos_l_star_computed = 0;
	this->max_c_star_rank = this->nchoosek;
	this->min_c_star_rank = 0;
	this->num_in_c_star = this->nchoosek;
	/* TODO: Make sure these things are recomputed properly! */
}

void MMSLightweightManager::recalculateAllCStarValues()
{
	clock_t start_clock = clock();

	this->recalculateLeftCStarInclusionExclusion();

	this->num_branchpos_l_star_computed = this->num_branchpos;

	this->seconds_in_l_star = this->seconds_in_l_star + ((double) (clock() - start_clock)) / (double) CLOCKS_PER_SEC;
}

void MMSLightweightManager::recalculateLeftCStarInclusionExclusion()
{
	int* nset = (int*) malloc(this->k * sizeof(int));
	int* nset_vals = (int*) malloc(this->k * sizeof(int));
	int* set = (int*) malloc(this->k * sizeof(int));
	int* neighbors = (int*) malloc(this->k * sizeof(int));
	int rank = 0;
	INDEX_TO_SET(this->n, this->k, rank, set);

	this->min_c_star_rank = -1;
	for ( int rank = 0; rank < this->nchoosek; rank++ )
	{
		if ( this->set_labels[rank] != LABEL_C_STAR )
		{
			this->left_c_star[rank] = 0;
			GET_SUCCESSOR(this->n, this->k, set);
			continue;
		}

		if ( this->min_c_star_rank < 0 )
		{
			this->min_c_star_rank = rank;
		}
		this->max_c_star_rank = rank + 1;

		/* RECOMPUTE! */
		int value = 1;
		int left_degree = this->getLeftNeighborRanks(set, neighbors);

		/* INCLUSION-EXCLUSION */
		/* start with single-sized neighbor sets */
		for ( int j = 0; j < left_degree; j++ )
		{
			int neighbor_rank = neighbors[j];
			if ( this->set_labels[neighbor_rank] == LABEL_C_STAR )
			{
				int tval = this->left_c_star[neighbor_rank];
				value += tval;
			}
		}

		/* correct for over/under counting */
		for ( int i = 2; i <= left_degree; i++ )
		{
			int num_sets = nChooseK(left_degree, i);

			INDEX_TO_SET(left_degree, i, 0, nset);
			for ( int nrank = 0; nrank < num_sets; nrank++ )
			{
				bool all_in_c_star = true;
				for ( int j = 0; j < i; j++ )
				{
					nset_vals[j] = neighbors[nset[j]];
					if ( this->set_labels[nset_vals[j]] != LABEL_C_STAR )
					{
						all_in_c_star = false;
					}
				}

				/* if any not in C*, then we definitely have 0 for this value */
				if ( all_in_c_star )
				{
					/* find the JOIN of these sets */
					int joinrank = this->getJoinRank(i, nset_vals);

					if ( this->set_labels[joinrank] == LABEL_C_STAR )
					{
						int tval = this->left_c_star[joinrank];

						if ( (i % 2) == 0 )
						{
							// Even sets were over counted
							value -= tval;
						}
						else
						{
							// Odd sets were under counted
							value += tval;
						}
					}
				}

				GET_SUCCESSOR(left_degree, i, nset);
			}
		}

		/* STORE VALUE */
		this->left_c_star[rank] = value;

		GET_SUCCESSOR(this->n, this->k, set);
	}

	free(set);
	free(neighbors);
	free(nset);
	free(nset_vals);
}

void MMSLightweightManager::recalculateLeftCStarIterative( int* new_set )
{
	/* go through our sets, join with new_set, and subtract previous left_c_star value. */
	int* set = (int*) malloc(this->k * sizeof(int));
	int* join_set = (int*) malloc(this->k * sizeof(int));
	this->max_c_star_rank = this->nchoosek + 1;

	int rank = this->nchoosek - 1;
	INDEX_TO_SET(this->n, this->k, rank, set);

	int j = 0;
	do
	{
		// this will be right the last time it is set
		this->min_c_star_rank = rank;

		if ( this->max_c_star_rank > this->nchoosek )
		{
			this->max_c_star_rank = rank + 1;
		}

		/* compute join and rank, starting where set changed */
		/* i = j ? (not if being colex/lex indifferent) */
		for ( int i = 0; i < this->k; i++ )
		{
			join_set[i] = set[i];

			if ( new_set[i] < set[i] )
			{
				join_set[i] = new_set[i];
			}
		}

		int join_rank = INDEX_OF_SET(this->n, this->k, join_set);
		int join_lstar = this->left_c_star[join_rank];

		if ( join_lstar > 0 )
		{
			// here's where the magic happens
			this->left_c_star[rank] = this->left_c_star[rank] - join_lstar;
		}

		j = GET_PREDECESSOR(this->n, this->k, set);
		rank--;
	}
	while ( rank >= 0 );

	free(set);
	free(join_set);
}

void MMSLightweightManager::initGenSets()
{
	this->num_forced_positive = 0;

	this->num_genneg = 0;
	this->max_genneg = 5000;
	this->genneg = (int*) malloc(this->max_genneg * sizeof(int*));

	this->num_branchneg = 0;
	this->max_branchneg = 500;
	this->branchneg = (int*) malloc(this->max_branchneg * sizeof(int));

	this->num_branchpos = 0;
	this->num_branchpos_l_star_computed = 0;
	this->max_branchpos = 500;
	this->branchpos = (int*) malloc(this->max_branchpos * sizeof(int));

	this->num_branches = 0;
	this->max_branches = 1000;
	this->branch_type = (int*) malloc(this->max_branches * sizeof(int));
	bzero(this->branch_type, this->max_branches * sizeof(int));

	this->snapshot();
}

void MMSLightweightManager::freeGenSets()
{
	if ( this->genneg != 0 )
	{
		free(this->genneg);
		this->genneg = 0;
	}

	if ( this->branchpos != 0 )
	{
		free(this->branchpos);
		this->branchpos = 0;
	}

	if ( this->branchneg != 0 )
	{
		free(this->branchneg);
		this->branchneg = 0;
	}

	if ( this->branch_type != 0 )
	{
		free(this->branch_type);
		this->branch_type = 0;
	}
}

void MMSLightweightManager::clearGenSets()
{
	this->num_genneg = 0;
	this->num_branchneg = 0;
	this->num_branchpos = 0;
	this->num_branches = 0;
	this->num_forced_positive = 0;

	while ( this->stack_branchneg.size() > 0 )
	{
		this->stack_branchneg.pop();
	}
	while ( this->stack_branchpos.size() > 0 )
	{
		this->stack_branchpos.pop();
	}
	while ( this->stack_branches.size() > 0 )
	{
		this->stack_branches.pop();
	}
}

void MMSLightweightManager::resetGenNeg()
{
	clock_t start_clock = clock();
	// first order of business
	this->num_genneg = 0;
	this->num_in_c_star = this->nchoosek - this->num_forced_positive;

	// we don't need to modify left_c_star, so ok.

	// second... clear all negative elements
	for ( int i = 0; i < this->nchoosek; i++ )
	{
		if ( this->set_labels[i] == LABEL_C_NEG )
		{
			this->set_labels[i] = LABEL_C_STAR;
		}
	}

	// now, we go through our negative branch sets and fix it!
	for ( int i = 0; i < this->num_branchneg; i++ )
	{
		this->markRightSetsNegative(this->branchneg[i]);
	}

	this->recalculateAllCStarValues();

	this->seconds_in_reset_genneg = this->seconds_in_reset_genneg
	        + ((double) (clock() - start_clock)) / (double) CLOCKS_PER_SEC;
}

void MMSLightweightManager::snapshot()
{
	this->snapshotLP();
	this->stack_branchneg.push(this->num_branchneg);
	this->stack_branchpos.push(this->num_branchpos);
	this->stack_branches.push(this->num_branches);
}

void MMSLightweightManager::rollback()
{
	this->clearLP();

	if ( this->stack_branchneg.size() > 0 )
	{
		int old_branchneg = this->num_branchneg;
		int old_branchpos = this->num_branchpos;
		this->num_branchneg = this->stack_branchneg.top();
		this->num_branchpos = this->stack_branchpos.top();
		this->num_branches = this->stack_branches.top();
		this->stack_branchneg.pop();
		this->stack_branchpos.pop();
		this->stack_branches.pop();

		this->clearSetLabels();
		this->clearCStarValues();

		// clear ALL of this out!
		this->num_forced_positive = 0;
		this->num_in_c_star = this->nchoosek;

		for ( int i = 0; i < this->num_branchpos; i++ )
		{
			this->markLeftSetsPositive(this->branchpos[i]);
		}

		for ( int i = 0; i < this->num_branchneg; i++ )
		{
			this->markRightSetsNegative(this->branchneg[i]);
		}

		this->recalculateAllCStarValues();
	}
}

int MMSLightweightManager::initSetLabels()
{
	this->set_labels = (int*) malloc(this->nchoosek * sizeof(int));
	bzero(this->set_labels, this->nchoosek * sizeof(int));

	this->min_c_star_rank = 0;

	this->max_c_star_rank = this->nchoosek;

	this->num_in_c_star = this->nchoosek;

	return this->nchoosek;
}

void MMSLightweightManager::freeSetLabels()
{
	if ( this->set_labels != 0 )
	{
		free(this->set_labels);
		this->set_labels = 0;
	}
}

void MMSLightweightManager::markLeftSetsPositive( int rank )
{
	if ( this->set_labels[rank] == LABEL_C_POS )
	{
		// no biggie
		return;
	}
	if ( this->set_labels[rank] == LABEL_C_NEG )
	{
		// BIGGIE
		printf("-- trying to mark %d positive, but it is negative!\n", rank);
		return;
	}

	clock_t start_clock = clock();

	int* cur_set = (int*) malloc(this->k * sizeof(int));
	int* neighbor_ranks = (int*) malloc(this->k * sizeof(int));

	std::queue<int> q;

	q.push(rank);
	this->set_labels[rank] = LABEL_C_POS;

	int number_converted = 0;

	while ( q.size() > 0 )
	{
		int cur_rank = q.front();
		q.pop();

		(this->num_in_c_star) = this->num_in_c_star - 1;
		this->num_forced_positive = this->num_forced_positive + 1;
		number_converted++;

		INDEX_TO_SET(this->n, this->k, cur_rank, cur_set);
		int left_degree = this->getLeftNeighborRanks(cur_set, neighbor_ranks);
		/* Walk over all left sets and also label them */
		for ( int i = 0; i < left_degree; i++ )
		{
			int neighbor_rank = neighbor_ranks[i];

			if ( this->set_labels[neighbor_rank] == LABEL_C_STAR )
			{
				q.push(neighbor_rank);
				this->set_labels[neighbor_rank] = LABEL_C_POS;
			}
		}
	}

	free(cur_set);
	free(neighbor_ranks);

	this->seconds_marking_left = this->seconds_marking_left
	        + ((double) (clock() - start_clock)) / (double) CLOCKS_PER_SEC;
}

void MMSLightweightManager::markRightSetsNegative( int rank )
{
	if ( this->set_labels[rank] != LABEL_C_STAR )
	{
		return;
	}

	clock_t start_clock = clock();

	int* cur_set = (int*) malloc(this->k * sizeof(int));
	int* neighbor_ranks = (int*) malloc(this->k * sizeof(int));

	std::queue<int> q;

	q.push(rank);
	this->set_labels[rank] = LABEL_C_NEG;

	int number_converted = 0;

	while ( q.size() > 0 )
	{
		int cur_rank = q.front();
		q.pop();

		(this->num_in_c_star) = this->num_in_c_star - 1;
		number_converted++;

		INDEX_TO_SET(this->n, this->k, cur_rank, cur_set);
		int right_deg = this->getRightNeighborRanks(cur_set, neighbor_ranks);

		/* Walk over all right sets and also label them */
		for ( int i = 0; i < right_deg; i++ )
		{
			int neighbor_rank = neighbor_ranks[i];

			if ( this->set_labels[neighbor_rank] == LABEL_C_STAR )
			{
				q.push(neighbor_rank);
				this->set_labels[neighbor_rank] = LABEL_C_NEG;
			}
			else if ( this->set_labels[neighbor_rank] == LABEL_C_POS )
			{
				printf("-- trying to set %d to negative while marking right sets... but it is positive?\n",
				       neighbor_rank);
			}

		}
	}

	free(cur_set);
	free(neighbor_ranks);

	this->seconds_marking_right = this->seconds_marking_right
	        + ((double) (clock() - start_clock)) / (double) CLOCKS_PER_SEC;
}

void MMSLightweightManager::branchPositiveSet( int rank, int type )
{
	if ( this->set_labels[rank] == LABEL_C_POS )
	{
		/* already marked positive */
		return;
	}
	else if ( this->set_labels[rank] == LABEL_C_NEG )
	{
		/* Big problem! Trying to mark a positive set to the right of a negative set*/
		/* Since we propagate negative sets, this should never happen */
		printf("-- Trying to BRANCH %10X positive but it is negative!\n", rank);
		return;
	}

	// do this in the marking

	// add to branchpos
	if ( this->num_branchpos >= this->max_branchpos )
	{
		this->max_branchpos = this->max_branchpos + 100;
		this->branchpos = (int*) realloc(this->branchpos, this->max_branchpos * sizeof(int));
	}
	this->branchpos[this->num_branchpos] = rank;
	this->num_branchpos = this->num_branchpos + 1;

	if ( this->num_branches >= this->max_branches )
	{
		this->max_branches = this->max_branches + 200;
		this->branch_type = (int*) realloc(this->branch_type, this->max_branches * sizeof(int));
	}
	this->branch_type[this->num_branches] = abs(type);
	this->num_branches = this->num_branches + 1;

	this->markLeftSetsPositive(rank);

	this->recalculateAllCStarValues();
}

void MMSLightweightManager::branchNegativeSet( int rank, int type )
{
	if ( this->set_labels[rank] == LABEL_C_NEG )
	{
		/* already marked negative */
		return;
	}
	else if ( this->set_labels[rank] == LABEL_C_POS )
	{
		/* Big problem! Trying to mark a negative set to the left of a positive set*/
		/* Since we propagate positive sets, this should never happen */
		printf("-- Trying to BRANCH %10X negative but it is positive!\n\t", rank);

		int* set = (int*) malloc(this->k * sizeof(int));
		INDEX_TO_SET(n, this->k, rank, set);
		for ( int j = 0; j < this->k; j++ )
		{
			printf("%2d ", set[j]);
		}
		printf("\n");
		free(set);
		return;
	}

	// add to branchneg
	if ( this->num_branchneg >= this->max_branchneg )
	{
		this->max_branchneg = this->max_branchneg + 100;
		this->branchneg = (int*) realloc(this->branchneg, this->max_branchneg * sizeof(int));
	}
	this->branchneg[this->num_branchneg] = rank;
	this->num_branchneg = this->num_branchneg + 1;

	if ( this->num_branches >= this->max_branches )
	{
		this->max_branches = this->max_branches + 200;
		this->branch_type = (int*) realloc(this->branch_type, this->max_branches * sizeof(int));
	}

	this->branch_type[this->num_branches] = -abs(type);
	this->num_branches = this->num_branches + 1;

//	int* cur_set = (int*) malloc(this->k * sizeof(int));
//	INDEX_TO_SET(this->n, this->k, rank, cur_set);
//	this->addNegativeBranchConstraint(cur_set);
//	free(cur_set);

	this->markRightSetsNegative(rank);

}

void MMSLightweightManager::propagateNegativeSet( int rank )
{
	if ( this->set_labels[rank] == LABEL_C_NEG )
	{
		/* already marked negative */
		return;
	}
	else if ( this->set_labels[rank] == LABEL_C_POS )
	{
		/* Big problem! Trying to mark a negative set to the left of a positive set*/
		/* Since we propagate positive sets, this should never happen */
		printf("-- Trying to PROPAGATE %10d negative but it is positive!\n\t", rank);

		int* set = (int*) malloc(this->k * sizeof(int));
		INDEX_TO_SET(n, this->k, rank, set);
		for ( int j = 0; j < this->k; j++ )
		{
			printf("%2d ", set[j]);
		}
		printf("\n");
		free(set);
		return;
	}

	// add to genneg
	if ( this->num_genneg >= this->max_genneg )
	{
		this->max_genneg = this->max_genneg + 100;
		this->genneg = (int*) realloc(this->genneg, this->max_genneg * sizeof(int));
	}

	this->genneg[this->num_genneg] = rank;
	this->num_genneg = this->num_genneg + 1;

//	int* cur_set = (int*) malloc(this->k * sizeof(int));
//	INDEX_TO_SET(this->n, this->k, rank, cur_set);
//	this->addNegativePropagationConstraint(cur_set);
//	free(cur_set);

	this->markRightSetsNegative(rank);

}

void MMSLightweightManager::clearSetLabels()
{
	bzero(this->set_labels, this->nchoosek * sizeof(int));
	this->min_c_star_rank = 0;
	this->max_c_star_rank = this->nchoosek;
	this->num_in_c_star = this->nchoosek;
	this->num_forced_positive = 0;
}

void MMSLightweightManager::initAll()
{
	this->found_solution = false;
	this->found_solution_at = -1;

	this->initSetLabels();
	this->initLeftShift();
	this->initCStarValues();
	this->initGenSets();
	this->initLP();

	if ( this->root != 0 )
	{
		delete this->root;
	}
	this->root = new MMSLWNode(0);
}

void MMSLightweightManager::freeAll()
{
	this->freeLeftShift();
	this->freeCStarValues();
	this->freeGenSets();
	this->num_forced_positive = 0;
	this->freeSetLabels();
	this->freeLP();
}

int MMSLightweightManager::getRandomSetInCStar()
{
	int N = this->num_in_c_star;

	if ( N == 0 )
	{
		printf("-- C STAR IS EMPTY!\n");
		return -1;
	}

	int random_N = rand();

	int unif_dist_size = N;
	for ( int i = 1; i <= this->num_steps_in_distribution; i++ )
	{
		unif_dist_size += (i * N) / (this->num_steps_in_distribution);
	}

	for ( int i = this->num_steps_in_distribution; i >= 1; i-- )
	{
		random_N = random_N % unif_dist_size;
		unif_dist_size -= (i * N) / (this->num_steps_in_distribution);
	}

	int cstar_i = random_N % N;

	int num = 0;

	for ( int i = 0; i < this->nchoosek; i++ )
	{
		if ( this->set_labels[i] == LABEL_C_STAR )
		{
			if ( num == cstar_i )
			{
				return i;
			}
			num++;
		}
	}

	return -1;
}

bool MMSLightweightManager::propagate()
{
	bool result = true;

	bool updated = true;

	result = this->propagateGAC();

	if ( !result || this->num_forced_positive >= this->mms_target )
	{
		return false;
	}

	result = this->isLPFeasible();

	return result;
}

bool MMSLightweightManager::propagateGAC()
{
	clock_t start_clock = clock();
	/* Add negative sets if their positive values become too much */
	bool added_negative = false;

	int num_added = 0;

	added_negative = false;
	int* set = (int*) malloc(this->k * sizeof(int));
	int rank = this->min_c_star_rank;
	INDEX_TO_SET(this->n, this->k, rank, set);

	int num_changed = 0;
	for ( ; rank < this->max_c_star_rank; rank++ )
	{
		if ( this->set_labels[rank] == LABEL_C_STAR )
		{
			if ( set[0] == 0 && this->left_shift[rank] >= this->mms_target - this->gvals[this->n - this->k] )
			{
				num_changed++;
				added_negative = true;
				this->propagateNegativeSet(rank);
			}
			else if ( this->left_c_star[rank] + this->num_forced_positive >= this->mms_target )
			{
				num_changed++;
				added_negative = true;
				this->propagateNegativeSet(rank);
			}
		}

		GET_SUCCESSOR(this->n, this->k, set);
	}
	free(set);


	this->seconds_in_gac = this->seconds_in_gac + ((double) (clock() - start_clock)) / (double) CLOCKS_PER_SEC;
	return true;
}

bool MMSLightweightManager::propagatePositive()
{
	/* Add negative sets if their positive values become too much */
	bool added_positive = false;
	bool ever_added_positive = false;

	int num_added = 0;
	added_positive = false;

	int* set = (int*) malloc(this->k * sizeof(int));
	int rank = this->max_c_star_rank - 1;
	INDEX_TO_SET(this->n, this->k, rank, set);

	for ( ; rank >= this->min_c_star_rank; rank-- )
	{
		if ( this->set_labels[rank] == LABEL_C_STAR )
		{
			INDEX_TO_SET(this->n, this->k, rank, set);
			bool result = this->isLPFeasibleWithConstraint(set, -1);

			if ( !result )
			{
				ever_added_positive = true;
				added_positive = true;
				num_added++;

				this->branchPositiveSet(rank, 2);
			}
		}

		GET_PREDECESSOR(this->n, this->k, set);
	}
	free(set);

	if ( ever_added_positive )
	{
		/* negative means update left c-star values */
		// happens on branch?
		this->recalculateAllCStarValues();
	}

	return true;
}

MMSLightweightManager::MMSLightweightManager( int n, int k ) :
		SearchManager()
{
	this->n = n;
	this->k = k;
	this->r = n % k;
	this->c = (n - this->r) / k;
	this->nchoosek = nChooseK(n, k);

	this->num_steps_in_distribution = 0; //k / 2;

	this->seconds_in_l_star = 0;
	this->seconds_in_lp = 0;
	this->seconds_in_stochastic = 0;
	this->seconds_in_gac = 0;
	this->seconds_in_reset_genneg = 0;
	this->seconds_marking_left = 0;
	this->seconds_marking_right = 0;
	this->seconds_in_lp_reset = 0;

	this->use_branchless = false;
	this->use_stochastic = false;
	this->done_branchless_or_stochastic = false;

	this->gvals = (int*) malloc(n * sizeof(int));

	bzero(this->gvals, n * sizeof(int));
	for ( int i = k; i < n; i++ )
	{
		int m = i - (i % k);
		this->gvals[i] = nChooseK(m - 1, k - 1);
	}
	this->mms_target = nChooseK(n - 1, k - 1);

	this->Baranyai_value = nChooseK((this->c - 1) * k - 1, k - 1);
	this->num_forced_positive = 0;
	this->propagation_mode = PROPAGATE_GAC;
	this->propagate_positive = false;
	this->root = 0;

	this->initAll();
}

/**
 * Default constructor
 */
/**
 * Destructor
 */
MMSLightweightManager::~MMSLightweightManager()
{
	this->freeAll();
	free(this->gvals);
}

/**
 * pushNext -- deepen the search to the next child
 * 	of the current node.
 *
 * @return the label for the new node. -1 if none.
 */
LONG_T MMSLightweightManager::pushNext()
{

	/* get 'top' node */
	SearchNode* parent = 0;
	if ( this->stack.size() > 0 )
	{
		parent = (SearchNode*) this->stack.back();
	}
	else
	{
		parent = (SearchNode*) this->root;
	}

	if ( parent->curChild >= 1 )
	{
		return -1;
	}

	LONG_T child = parent->curChild + 1;

	return this->pushTo(child);
}

/**
 * pushTo -- deepen the search to the specified child
 * 	of the current node.
 *
 * @param child the specified label for the new node
 * @return the label for the new node. -1 if none, or failed.
 */
LONG_T MMSLightweightManager::pushTo( LONG_T child )
{
	/* get 'top' node */
	MMSLWNode* parent = 0;
	if ( this->stack.size() > 0 )
	{
		parent = (MMSLWNode*) this->stack.back();
	}
	else
	{
		parent = (MMSLWNode*) this->root;
	}
	parent->curChild = child;

	// ok... start over...
	bool result = false;
	// not loading job, generating jobs, or loading a partial job...
	if ( this->deepeningMode == 0 && parent->tested_stochastic == false )
	{
		if ( this->use_stochastic )
		{
			result = this->doStochasticSearch();
		}
//
//		// time for branchless/stochastic
//		if ( this->use_branchless )
//		{
//			result = this->doBranchlessSearch();
//		}

		parent->tested_stochastic = true;

//		if ( !result )
//		{
//			// already printed..
//			return -1;
//		}
	}

	this->snapshot();
	this->push_since_pop = true;
	this->resetGenNeg();
	this->propagateGAC();

	if ( parent->branch_rank < 0 )
	{
		if ( (child & 0xFFFFFFFF00) != 0 )
		{
			// pull out the rank of the previously-selected set
			parent->branch_rank = (child >> 8) - 1;
		}
		else
		{
			parent->branch_rank = this->getRandomSetInCStar();
		}
	}

	if ( parent->branch_rank < 0 )
	{
		this->rollback();
		return -1;
	}

	if ( (child & 0xF) == 0 )
	{
		this->resetGenNeg();
		this->clearPropagationConstraints();
		this->branchNegativeSet(parent->branch_rank, -1);

		int* set = (int*) malloc(this->k * sizeof(int));
		INDEX_TO_SET(this->n, this->k, parent->branch_rank, set);
		this->addNegativeBranchConstraint(set);
		free(set);
	}
	else if ( (child & 0xF) == 1 )
	{
		this->resetGenNeg();
		this->clearPropagationConstraints();
		this->branchPositiveSet(parent->branch_rank, 1);
		int* set = (int*) malloc(this->k * sizeof(int));
		INDEX_TO_SET(this->n, this->k, parent->branch_rank, set);
		this->addPositiveBranchConstraint(set);
		free(set);
	}
	else
	{
		this->rollback();
		/* incorrect! */
		return -1;
	}

	child = child | (((LONG_T) (parent->branch_rank + 1)) << 8);

//	printf("--/pushTo(%6llX)\n", child);

	MMSLWNode* cnode = new MMSLWNode(child);

	this->stack.push_back(cnode);

	return child;
}

/**
 * pop -- remove the current node and move up the tree.
 *
 * @return the label of the node after the pop.
 * 		This return value is used for validation purposes
 * 		to check proper implementation of push*() and pop().
 */
LONG_T MMSLightweightManager::pop()
{

	if ( this->stack.size() > 0 )
	{
		SearchNode* node = this->stack.back();
		this->stack.pop_back();
		LONG_T child = node->label;
		delete node;

//		for ( int i = 0; i <= this->searchDepth; i++ )
//		{
//			printf(".");
//		}
//		printf("pop(%6llX) (tree)\n", child);

		if ( this->found_solution_at >= this->searchDepth )
		{
			this->found_solution_at = -1;
		}

		this->push_since_pop = false;

		this->rollback();

		return child;
	}
	else
	{
//		for ( int i = 0; i <= this->searchDepth; i++ )
//		{
//			printf(".");
//		}
//		printf("pop(???) (tree)\n");
		this->clearAll();
		return -1;
	}
}

/**
 * prune -- Perform a check to see if this node should be pruned.
 *
 * @return 0 if no prune should occur, 1 if prune should occur.
 */
int MMSLightweightManager::prune()
{
//	printf("--prune()\n");

	this->clearCStarValues();
	this->resetGenNeg();
	this->recalculateAllCStarValues();

	if ( this->num_in_c_star == 0 || this->num_forced_positive >= this->mms_target )
	{
		this->printBranchlessData(0, -1, -1, true);
		return 1;
	}

	bool result = false;

	result = this->propagate();

	if ( !result )
	{
		// report the END of this line!
		this->printBranchlessData(0, -1, -1, true);
		return 1;
	}

	if ( this->num_in_c_star == 0 || this->num_forced_positive >= this->mms_target )
	{
		this->printBranchlessData(0, -1, -1, true);
		return 1;
	}

	return 0;
}

/**
 * isSolution -- Perform a check to see if a solution exists
 * 		at this point.
 *
 *  By "Solution" we of course mean a "counterexample" to the conjecture.
 *  Test the LP feasible point for a counterexample.
 *
 * @return 0 if no solution is found, 1 if a solution is found.
 */
int MMSLightweightManager::isSolution()
{
	bool result = this->isLPOptimalACounterexample();

	if ( result )
	{
		if ( this->found_solution_at < 0 )
		{
			this->found_solution = true;
			this->found_solution_at = this->searchDepth;
//			printf("-- found solution at depth %d\n", this->searchDepth);
		}

		return 1;
	}

	return 0;
}

/**
 * writeSolution -- create a buffer that contains a
 * 	description of the solution.
 *
 * Write the counterexample, as well as the number of non-negative k-sets and how that violates expectation
 *
 * @return a string of data allocated with malloc().
 * 	It will be deleted using free().
 */
char* MMSLightweightManager::writeSolution()
{
	this->printLPOptimal();
	printf("T MIN NUMBER_OF_SETS %d VERSUS %d\n", this->sol_pos_sets, nChooseK(this->n - 1, this->k - 1));

	return 0;
}

/**
 * writeStatistics -- create a buffer that contains a
 * 	description of the solution.
 *
 * Statistics take the following format in each line:
 *
 * T [TYPE] [ID] [VALUE]
 *
 * @return a string of data allocated with malloc().
 * 	It will be deleted using free().
 */
/**
 * ClearAll will restart the search from scratch, but won't reallocate.
 * It just clears the data to zero.
 */
char* MMSLightweightManager::writeStatistics()
{
	/* TODO: collect and write statistics */
	if ( this->found_solution == false )
	{
		printf("T MIN NUMBER_OF_SETS %d VERSUS %d\n", this->mms_target, nChooseK(this->n - 1, this->k - 1));
	}
	else
	{
		printf("T MIN NUMBER_OF_SETS %d VERSUS %d\n", this->sol_pos_sets, nChooseK(this->n - 1, this->k - 1));
	}

	printf("T SUM TIME_IN_LP    %5.5lf\n", this->seconds_in_lp);
	printf("T SUM TIME_RESET_LP %5.5lf\n", this->seconds_in_lp_reset);
	printf("T SUM TIME_IN_GAC   %5.5lf\n", this->seconds_in_gac);
	printf("T SUM TIME_IN_LSTAR %5.5lf\n", this->seconds_in_l_star);
	printf("T SUM TIME_IN_STOCH %5.5lf\n", this->seconds_in_stochastic);
	printf("T SUM TIME_IN_RESET %5.5lf\n", this->seconds_in_reset_genneg);
	printf("T SUM TIME_M_LEFT   %5.5lf\n", this->seconds_marking_left);
	printf("T SUM TIME_M_RIGHT  %5.5lf\n", this->seconds_marking_right);

	return super::writeStatistics();
}

void MMSLightweightManager::clearAll()
{
	this->clearCStarValues();
	this->clearGenSets();
	this->clearSetLabels();
	this->clearLP();

	this->found_solution = false;
	this->found_solution_at = -1;
	this->num_forced_positive = 0;

//	this->seconds_in_l_star = 0;
//	this->seconds_in_lp = 0;
//	this->seconds_in_stochastic = 0;
//	this->seconds_in_gac = 0;

	this->done_branchless_or_stochastic = false;

	if ( this->strong_mode )
	{
		/* we need to add the set 0, n-k+1, ..., n-1 to cut_negative */
		int* set = (int*) malloc(this->k * sizeof(int));

		set[0] = 0;
		for ( int i = 1; i < this->k; i++ )
		{
			set[i] = this->n - this->k + i;
		}
		int index = INDEX_OF_SET(this->n, this->k, set);
		this->branchNegativeSet(index, -1);

		free(set);
		set = 0;
	}
}

/**
 * setTarget
 *
 * Change the target number of non-negative k-sums to avoid.
 *
 * Useful for computing g(n,k) when it is strictly less than (n-1 choose k-1).
 */
void MMSLightweightManager::setTarget( int target )
{
	this->mms_target = target;
}

/**
 * setGValue
 *
 * Store a previously-computed value of g(m,k) for some other m < n.
 */
void MMSLightweightManager::setGValue( int m, int value )
{
	if ( m < this->k || m >= this->n )
	{
		return;
	}

	this->gvals[m] = value;
}

/**
 * setStrongMode
 *
 * Change whether or not to test for strong pair (n,k).
 */
void MMSLightweightManager::setStrongMode( bool mode )
{
	this->strong_mode = mode;
}

/**
 * doBranchlessSearch
 *
 *
 */
bool MMSLightweightManager::doBranchlessSearch()
{
	time_t start_time = time(NULL);
	clock_t start_clock = clock();

	this->num_branchpos_before_stochastic = this->num_branchpos;

	bool result = true;
	long long int num_samples = 0;
	while ( result )
	{
		this->resetGenNeg();
		result = this->propagateGAC();

		if ( this->num_in_c_star == 0 && this->num_forced_positive >= this->mms_target )
		{
			this->writeCompleteJob(stdout);
			this->printBranchlessData(num_samples, start_time, start_clock, true);
			return false;
		}

		if ( result )
		{
			result = this->isLPFeasible();

			if ( this->isLPOptimalACounterexample() )
			{
				this->writeCompleteJob(stdout);
				this->printLPOptimal();
				this->printBranchlessData(num_samples, start_time, start_clock, false);
				return true;
			}
		}

		if ( !result )
		{
			/* AHA! */
			this->writeCompleteJob(stdout);
			this->printBranchlessData(num_samples, start_time, start_clock, true);
			return false;
		}

		result = this->propagatePositive();

		if ( this->num_forced_positive >= this->mms_target )
		{
			this->writeCompleteJob(stdout);
			this->printBranchlessData(num_samples, start_time, start_clock, true);
			return false;
		}
	}

// write it as PARTIAL for use in supercomputer!
	this->writePartialJob(stdout);
	this->printBranchlessData(num_samples, start_time, start_clock, false);
	return true;
}

/**
 * doStochasticSearch
 *
 * Do ALL negative propagations, then randomly select sets to check if they make the LP infeasible
 * when negative. Make these sets positive, propagate negatives, and repeat!
 *
 */
bool MMSLightweightManager::doStochasticSearch()
{
	time_t start_time = time(NULL);
	clock_t start_clock = clock();

	this->num_branchpos_before_stochastic = this->num_branchpos;
	this->num_branchneg_before_stochastic = this->num_branchneg;

	srand((unsigned) start_time);

	int samples_since_success = 0;

	bool result = true;
	long long int num_samples = 0;

	while ( time(NULL) < start_time + this->sample_time && clock() < start_clock + CLOCKS_PER_SEC * this->sample_time )
	{
		this->resetGenNeg();
		result = this->propagateGAC();

		if ( !result || (this->num_in_c_star == 0 && this->num_forced_positive >= this->mms_target) )
		{
//			printf("-- ending since propagation said NO!\n");
			this->printBranchlessData(num_samples, start_time, start_clock, true);
			return false;
		}

		if ( result )
		{
			result = this->isLPFeasible();

			if ( this->isLPOptimalACounterexample() )
			{
				this->printLPOptimal();
				this->printBranchlessData(num_samples, start_time, start_clock, false);
				return true;
			}
			else if ( this->num_in_c_star == 0 )
			{
//				printf("--we have nothing in cstar, the number of sets is right, but no counterexample??\n");
			}
		}

		if ( !result )
		{
			/* AHA! */
//			printf("-- ending since LP is infeasible!\n");
			this->printBranchlessData(num_samples, start_time, start_clock, true);
			return false;
		}

		bool found_pos = false;
		while ( !found_pos && time(NULL) < start_time + this->sample_time
		        && clock() < start_clock + CLOCKS_PER_SEC * this->sample_time )
		{
			num_samples++;
			samples_since_success++;

			// we find a random rank...
			int rank = this->getRandomSetInCStar();

			int* set = (int*) malloc(this->k * sizeof(int));
			INDEX_TO_SET(this->n, this->k, rank, set);

			bool inner_result = this->isLPFeasibleWithConstraint(set, -1);

			if ( !inner_result )
			{
				samples_since_success = 0;

				this->branchPositiveSet(rank, 2);

				found_pos = true;
				// will be snapshot on the way back

				if ( this->num_forced_positive >= this->mms_target )
				{
//					printf("-- ending since too many positive sets!\n");
					this->printBranchlessData(num_samples, start_time, start_clock, true);

					free(set);
					return false;
				}
			}
			else
			{
				bool inner_result = this->isLPFeasibleWithConstraint(set, 1);

				if ( !inner_result )
				{
					samples_since_success = 0;
					this->resetGenNeg();

					this->branchNegativeSet(rank, -2);

					found_pos = true;
					// will be snapshot on the way back

					if ( this->num_forced_positive >= this->mms_target )
					{
						//					printf("-- ending since too many positive sets!\n");
						this->printBranchlessData(num_samples, start_time, start_clock, true);

						free(set);
						return false;
					}
				}
			}

			free(set);

			if ( samples_since_success > this->num_failed_samples_to_halt )
			{
				// stop for now, continue branching!
				this->printBranchlessData(num_samples, start_time, start_clock, false);
//				printf("-- too many samples, back to branching.\n");

				this->resetGenNeg();
				return true;
			}
		}
	}

// write it as PARTIAL for use in supercomputer!
//	printf("\tQuit early!\n");
	this->printBranchlessData(num_samples, start_time, start_clock, false);
	return true;
}

void MMSLightweightManager::useBranchlessSearch( bool branchless )
{
	this->use_branchless = branchless;
}

void MMSLightweightManager::useStochasticSearch( bool stochastic )
{
	this->use_stochastic = stochastic;
}

void MMSLightweightManager::setNumSamplesToHalt( int num )
{
	this->num_failed_samples_to_halt = num;
}

void MMSLightweightManager::setSampleTime( int sample_time )
{
	this->sample_time = sample_time;
}

void MMSLightweightManager::setDistributionSteps( int steps )
{
	this->num_steps_in_distribution = steps;
}

void MMSLightweightManager::printBranchlessData( int num_samples, time_t starttime, clock_t startclock, bool done )
{
	// so we don't forget:

	if ( num_samples > 0 )
	{
		printf("T SUM NUM_SAMPLES %lld\n", num_samples);
		time_t end_time = time(NULL);
		if ( end_time - starttime > 120 )
		{
			printf("T SUM STOCHASTIC_SEARCH_TIME %ld\n\n", (time(NULL) - starttime));
		}
		else
		{
			printf("T SUM STOCHASTIC_SEARCH_TIME %2.4lf\n\n",
			       ((double) (clock() - startclock)) / (double) CLOCKS_PER_SEC);
		}

		this->seconds_in_stochastic = this->seconds_in_stochastic
		        + ((double) (clock() - startclock)) / (double) CLOCKS_PER_SEC;
	}

	if ( done )
	{
		// now, print the sets!
		int cur_pos = 0;
		int cur_neg = 0;
		printf("-------------------------------------------\n");
		int* set = (int*) malloc(this->k * sizeof(int));
		for ( int i = 0; i < this->num_branches; i++ )
		{
			if ( this->branch_type[i] > 0 )
			{
				if ( this->branch_type[i] == 1 )
				{
					printf("\nXBP ");
				}
				else if ( this->branch_type[i] == 2 )
				{
					printf("XPP ");
				}
				else
				{
					printf("X?P "); // shouldn't happen
				}

				int rank = this->branchpos[cur_pos];
				INDEX_TO_SET(this->n, this->k, rank, set);

				for ( int j = 0; j < this->k; j++ )
				{
					printf("%02d ", set[j]);
				}
				printf("\n");

				cur_pos++;
			}
			else
			{
				if ( this->branch_type[i] == -1 )
				{
					printf("\nXBN ");
				}
				else if ( this->branch_type[i] == -2 )
				{
					printf("XPN ");
				}
				else
				{
					printf("X?N "); // shouldn't happen
				}

				int rank = this->branchneg[cur_neg];
				INDEX_TO_SET(this->n, this->k, rank, set);

				for ( int j = 0; j < this->k; j++ )
				{
					printf("%02d ", set[j]);
				}
				printf("\n");

				cur_neg++;
			}
		}
		printf("\nSTOP\n");
		printf("-------------------------------------------\n");
		free(set);
	}
	fflush(stdout);
}

void MMSLightweightManager::printStatus()
{
	printf("BranchPos:   %5d\tBranchNeg:  %5d\tGenNeg: %5d\n", this->num_branchpos, this->num_branchneg,
	       this->num_genneg);
	printf("NumPositive: %5d\tNumInCStar: %5d\n", this->num_forced_positive, this->num_in_c_star);
//	printf("num_constraints: %5d\n\n", this->num_constraints);
}
