/*
 * MMSBranchingManager.hpp
 *
 * The MMSBranchingManager class is a TreeSearch implementation of a branching
 * procedure to verify the MMS Conjecture for a pair (n,k) using linear programming.
 *
 *  Created on: Jul 19, 2012
 *      Author: stolee
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stack>
#include "Set.hpp"
#include "SearchManager.hpp"

#define INDEX_OF_SET(N,K,SET) indexOfSet(K,SET)
#define INDEX_TO_SET(N,K,INDEX,SET) indexToSet(K,INDEX,SET)
#define GET_SUCCESSOR(N,K,SET) getSuccessor(K,SET)
#define GET_PREDECESSOR(N,K,SET) getPredecessor(K,SET)

#ifdef CPLEX
extern "C"
{
#include "ilcplex/cplex.h"
}
#endif

#ifdef GLPK
extern "C"
{
#include <glpk.h>
}
#endif

#ifdef COINOR

#endif

#ifdef GUROBI

#endif

class MMSLWNode: public SearchNode
{
	public:
		MMSLWNode( LONG_T label );
		virtual ~MMSLWNode();
		int branch_rank;
		bool tested_stochastic;
};

class MMSLightweightManager: public SearchManager
{
	public:
		typedef SearchManager super;

		/**
		 * n and k are the parameters of the conjecture we are testing.
		 * c and r are such that n = ck + r (r in 0...k-1)
		 */
		int n, k, c, r;
		int nchoosek;
		double sample_time;

		/**
		 * gvals[m] stores g(m,k).
		 *
		 * Initialized using Baranyai.
		 */
		int* gvals;

		/**
		 * Store {n-1 choose k-1}, or whatever target we are looking for.
		 */
		int mms_target;

		/**
		 * Store { (c-1)k - 1 choose k-1 } where n = ck+r.
		 */
		int Baranyai_value;

		bool strong_mode;

		/* used for pruning AFTER a solution was found, avoids duplicates, hopefully */
		bool found_solution;
		bool push_since_pop;
		int found_solution_at;

		/**
		 * getLeftNeighborRanks
		 *
		 * @param set - the set to start at.
		 * @param ranks - an array of at least k entries to fill with ranks
		 * @return number of ranks actually used (could be less than k)
		 */
		int getLeftNeighborRanks( int* set, int* ranks );

		/**
		 * getRightNeighborRanks
		 *
		 * @param set - the set to start at.
		 * @param ranks - an array of at least k entries to fill with ranks
		 * @return number of ranks actually used (could be less than k)
		 */
		int getRightNeighborRanks( int* set, int* ranks );

		int getMeetRank( int num_sets, int* set_ranks );
		int getJoinRank( int num_sets, int* set_ranks );

		/**
		 * The left-shift values stored in a table.
		 */
		int* left_shift;
		int initLeftShift();
		void freeLeftShift();

		int* left_c_star;

		int initCStarValues();
		void freeCStarValues();
		void clearCStarValues();

		void recalculateLeftCStarInclusionExclusion();
		void recalculateLeftCStarIterative( int* new_set );

		int num_forced_positive;

		/***
		 * Store the list of sets that generate C^+ and C^-
		 *
		 * branchpos : sets selected to be nonnegative
		 * branchneg : sets selected to be negative
		 * genneg 	 : sets determined to be negative by propagation
		 */
		int num_branches;
		int max_branches;
		int* branch_type; // 1 for positive, -1 for negative.
		int num_branchpos;
		int num_branchpos_before_stochastic;
		int num_branchneg_before_stochastic;
		int num_branchpos_l_star_computed; // how much is lstar updated to?
		int max_branchpos;
		int* branchpos;
		int num_branchneg;
		int max_branchneg;
		int* branchneg;
		int num_genneg;
		int max_genneg;
		int* genneg;
		void initGenSets();
		void freeGenSets();
		void clearGenSets();
		void propagateNegativeSet( int rank );
		std::stack<int> stack_branchpos;
		std::stack<int> stack_branchneg;
		std::stack<int> stack_branches;

		/**
		 * The C^+, C^- and C^* sets contain the current sets of positively-selected
		 * k-sets, negatively-selected k-sets, and currently-open k-sets.
		 *
		 * These take different values within the set_labels array.
		 *
		 * 0 - C^*
		 * 1 - C^+
		 * 2 - C^-
		 */
#define LABEL_C_STAR 0
#define LABEL_C_POS 1
#define LABEL_C_NEG 2
		int* set_labels;
		int min_c_star_rank;
		int max_c_star_rank;
		int num_in_c_star;
		int initSetLabels();
		void freeSetLabels();
		void snapshotSetLabels();
		void rollbackSetLabels();
		void clearSetLabels();
		void resetAllLabels(); // clear all labels, since we are probably popping a positive set
		void markLeftSetsPositive( int rank ); // for resetting when popping
		void markRightSetsNegative( int rank ); // for resetting when popping

#ifdef CPLEX
		/**
		 * The LP instance in terms of CPLEX
		 */
		int num_constraints;
		int* isolution;
		CPXLPptr lpx;
		int num_positive;
		int num_sure_positive;
		int num_integer_positive;
		CPXENVptr env;
#endif

		int sol_pos_sets;

#ifdef GLPK
		/**
		 * The LP instance in terms of GLPK
		 */
		// TODO: add GLPK THINGS
		glp_prob *lp;
		glp_smcp* parm;
		int num_rows;
		std::stack<int> stack_lp_rows;
#endif

#ifdef COINOR
		/**
		 * The LP instance in terms of COINOR
		 */
		// TODO: add COINOR THINGS
#endif

#ifdef GUROBI
		/**
		 * The LP instance in terms of GUROBI
		 */
		// TODO: add GUROBI THINGS
#endif

		double* solution;

		void initLP();
		void freeLP();
		void clearLP();
		void rebuildLP();
		int start_propagation_constraints; // also ends branching constraints
		int start_temporary_constraints; // also ends propagation constraints
		void removeTopBranchConstraint();
		void clearPropagationConstraints();
		bool isLPOptimalACounterexample();
		void printLPOptimal();
		void rollbackLP();
		void snapshotLP();
		std::stack<int> stack_num_constraints;

		/**
		 * Handle all of the memory things.
		 */
		void initAll();
		void freeAll();

		/**
		 * Select a set to branch upon.
		 */
		// RANDOM BRANCH (will change the job description to recover random branch)
		int num_steps_in_distribution; // effects how left-heavy the search goes
		int getRandomSetInCStar();

		/**
		 * given the current situation, determine if some sets _must_ be negative in a counterexample.
		 */
#define PROPAGATE_NONE 0
#define PROPAGATE_GAC 1
		char propagation_mode;
		bool propagate_positive;
		bool propagateGAC();
		bool propagatePositive();

		bool use_branchless;
		bool use_stochastic;
		bool done_branchless_or_stochastic;
		int num_failed_samples_to_halt;

	public:
		/**
		 * Default constructor
		 */
		MMSLightweightManager( int n, int k );

		/**
		 * Destructor
		 */
		virtual ~MMSLightweightManager();

		/**
		 * pushNext -- deepen the search to the next child
		 * 	of the current node.
		 *
		 * @return the label for the new node. -1 if none.
		 */
		virtual LONG_T pushNext();

		/**
		 * pushTo -- deepen the search to the specified child
		 * 	of the current node.
		 *
		 * @param child the specified label for the new node
		 * @return the label for the new node. -1 if none, or failed.
		 */
		virtual LONG_T pushTo( LONG_T child );

		/**
		 * pop -- remove the current node and move up the tree.
		 *
		 * To do this, we simply remove the propagated negative sets and the most recent branch set.
		 *
		 * @return the label of the node after the pop.
		 * 		This return value is used for validation purposes
		 * 		to check proper implementation of push*() and pop().
		 */
		virtual LONG_T pop();

		/**
		 * prune -- Perform a check to see if this node should be pruned.
		 *
		 * @return 0 if no prune should occur, 1 if prune should occur.
		 */
		virtual int prune();

		/**
		 * isSolution -- Perform a check to see if a solution exists
		 * 		at this point.
		 *
		 *  By "Solution" we of course mean a "counterexample" to the conjecture.
		 *  Test the LP feasible point for a counterexample.
		 *
		 * @return 0 if no solution is found, 1 if a solution is found.
		 */
		int isSolution();

		// LP METHODS
		bool isLPFeasible();
		bool isLPFeasibleWithConstraint( int* set, int direction );

		/**
		 * writeSolution -- create a buffer that contains a
		 * 	description of the solution.
		 *
		 * Write the counterexample, as well as the number of non-negative k-sets and how that violates expectation
		 *
		 * @return a string of data allocated with malloc().
		 * 	It will be deleted using free().
		 */
		virtual char* writeSolution();

		/**
		 * writeStatistics -- create a buffer that contains a
		 * 	description of the solution.
		 *
		 * Statistics take the following format in each line:
		 *
		 * T [TYPE] [ID] [VALUE]
		 *
		 * @return a string of data allocated with malloc().
		 * 	It will be deleted using free().
		 */
		virtual char* writeStatistics();

		/**
		 * ClearAll will restart the search from scratch, but won't reallocate.
		 * It just clears the data to zero.
		 */
		void clearAll();

		void setDistributionSteps( int num_steps );
		void setStrongMode( bool mode );
		void setSampleTime( int sample_time );

		/**
		 * setTarget
		 *
		 * Change the target number of non-negative k-sums to avoid.
		 *
		 * Useful for computing g(n,k) when it is strictly less than (n-1 choose k-1).
		 */
		void setTarget( int target );

		/**
		 * setGValue
		 *
		 * Store a previously-computed value of g(m,k) for some other m < n.
		 */
		void setGValue( int m, int value );

		/**
		 * Propagate.
		 *
		 * Return FALSE if infeasible!
		 */
		bool propagate();

		/* public for tests */
		void branchPositiveSet( int rank, int type );
		void branchNegativeSet( int rank, int type );
		void addPositiveBranchConstraint( int* set );
		void addNegativeBranchConstraint( int* set );
		void addNegativePropagationConstraint( int* set );
		void addNegativeTemporaryConstraint( int* set );
		void addPositiveTemporaryConstraint( int* set );
		void removeTemporaryConstraints();
		void snapshot();
		void rollback();
		void recalculateAllCStarValues();
		void resetGenNeg();

		/**
		 * doBranchlessSearch
		 *
		 * Do ALL negative propagations, then do ALL positive propagations.
		 * REPEAT.
		 *
		 * returns FALSE if it reaches a stable point with propagations to make.
		 */
		bool doBranchlessSearch();
		void useBranchlessSearch( bool branchless );

		/**
		 * doStochasticSearch
		 *
		 * Do ALL negative propagations, then randomly select sets to check if they make the LP infeasible
		 * when negative. Make these sets positive, propagate negatives, and repeat!
		 *
		 */
		bool doStochasticSearch();
		void useStochasticSearch( bool stochastic );
		void setNumSamplesToHalt( int num );

		void printBranchlessData( int num_samples, time_t starttime, clock_t startclock, bool done );

		void printStatus();

		/// PERFORMANCE TRACKING
		double seconds_in_l_star;
		double seconds_in_lp;
		double seconds_in_lp_reset;
		double seconds_in_stochastic;
		double seconds_in_gac;
		double seconds_in_reset_genneg;
		double seconds_marking_right;
		double seconds_marking_left;
};

