/***********************************************************

Copyright Derrick Stolee 2011.

 This file is part of SearchLib.

    SearchLib is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    SearchLib is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with SearchLib.  If not, see <http://www.gnu.org/licenses/>.

*************************************************************/

/*
 * ProgressionTable.cpp
 *
 *  Created on: May 15, 2011
 *      Author: derrickstolee
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "ProgressionTable.hpp"
#include "ProgressionConstraint.hpp"
#include "PseudoBackwardConstraint.hpp"
#include "PseudoBDomainConstraint.hpp"
#include "PseudoFBDomainConstraint.hpp"
#include "PseudoForwardConstraint.hpp"
#include "QuasiBackwardConstraint.hpp"
#include "QuasiBDomainConstraint.hpp"
#include "QuasiFBDomainConstraint.hpp"
#include "QuasiForwardConstraint.hpp"

int ProgressionTable::getMinPos()
{
	if ( this->sym_mode == NO_SYMMETRY )
	{
		return 0;
	}

	return -(this->maxN);
}

int ProgressionTable::getMaxPos()
{
	return this->maxN;
}

void ProgressionTable::initColors()
{
	this->colors = (char*) malloc(this->maxN);

	for ( int i = 0; i < this->maxN; i++ )
	{
		this->colors[i] = (char) UNKNOWN;
	}

	this->color_snapshots.push(0);
}

void ProgressionTable::cleanColors()
{
	if ( this->colors != 0 )
	{
		free(this->colors);
		this->colors = 0;
	}

	while ( this->color_augmentations.size() > 0 )
	{
		this->color_augmentations.pop();
	}

	while ( this->color_snapshots.size() > 0 )
	{
		this->color_snapshots.pop();
	}
}

void ProgressionTable::snapshotColors()
{
	this->color_snapshots.push(this->color_augmentations.size());
}

void ProgressionTable::rollbackColors()
{
	int to_size = this->color_snapshots.top();
	this->color_snapshots.pop();

	while ( this->color_augmentations.size() > to_size )
	{
		int j = this->color_augmentations.top();
		this->color_augmentations.pop();

		this->colors[j] = (char) UNKNOWN;

		if ( j < this->curN )
		{
			this->curN = j;
		}
	}
}

void ProgressionTable::initDomains()
{
	this->domains = (char*) malloc(this->maxN);
	this->neg_domains = (char*) malloc(this->maxN);
	this->domain_sizes = (char*) malloc(this->maxN);
	this->neg_domain_sizes = (char*) malloc(this->maxN);

	char fulldomain = 0;
	for ( int i = 0; i < this->numColors; i++ )
	{
		fulldomain = fulldomain | (1 << i);
	}

	for ( int i = 0; i < this->maxN; i++ )
	{
		this->domains[i] = fulldomain;
		this->neg_domains[i] = fulldomain;
		this->domain_sizes[i] = this->numColors;
		this->neg_domain_sizes[i] = this->numColors;
	}

	this->domain_snapshots.push(0);
	this->neg_domain_snapshots.push(0);
	this->domain_size_snapshots.push(0);
	this->neg_domain_size_snapshots.push(0);
}

void ProgressionTable::cleanDomains()
{
	if ( this->domains != 0 )
	{
		free(this->domains);
		this->domains = 0;
	}

	if ( this->neg_domains != 0 )
	{
		free(this->neg_domains);
		this->neg_domains = 0;
	}

	if ( this->domain_sizes != 0 )
	{
		free(this->domain_sizes);
		this->domain_sizes = 0;
	}

	if ( this->neg_domain_sizes != 0 )
	{
		free(this->neg_domain_sizes);
		this->neg_domain_sizes = 0;
	}

	while ( this->domain_augmentations.size() > 0 )
	{
		this->domain_augmentations.pop();
	}

	while ( this->domain_snapshots.size() > 0 )
	{
		this->domain_snapshots.pop();
	}

	while ( this->neg_domain_augmentations.size() > 0 )
	{
		this->neg_domain_augmentations.pop();
	}

	while ( this->neg_domain_snapshots.size() > 0 )
	{
		this->neg_domain_snapshots.pop();
	}

	while ( this->domain_size_augmentations.size() > 0 )
	{
		this->domain_size_augmentations.pop();
	}

	while ( this->domain_size_snapshots.size() > 0 )
	{
		this->domain_size_snapshots.pop();
	}

	while ( this->neg_domain_size_augmentations.size() > 0 )
	{
		this->neg_domain_size_augmentations.pop();
	}

	while ( this->neg_domain_size_snapshots.size() > 0 )
	{
		this->neg_domain_size_snapshots.pop();
	}
}

char ProgressionTable::getDomain(int pos)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		return 0;
	}

	if ( pos < 0 )
	{
		if ( this->sym_mode != NO_SYMMETRY )
		{
			pos = -1 - pos;
			return this->neg_domains[pos];
		}
		else
		{
			return 0;
		}
	}

	return this->domains[pos];
}

bool ProgressionTable::removeColorFromDomain(int pos, char color)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		printf("--[ProgressionTable::removeColorFromDomain(%d,%d)] Pos out of range.\n", pos, color);
		return false;
	}

	bool pos_in_kill_zone = this->isInKillRange(pos);

	if ( this->getColor(pos) != (char) UNKNOWN )
	{
		if ( this->getColor(pos) == color )
		{
			/* we can't remove this one! */
			return false || !pos_in_kill_zone;
		}
		else
		{
			return true;
		}
	}

	char mask = (1 << color);

	if ( pos < 0 )
	{
		if ( this->sym_mode == NO_SYMMETRY )
		{
			return false;
		}

		int t_pos = -1 - pos;

		int domain_val = this->neg_domains[t_pos];
		int domain_size = this->neg_domain_sizes[t_pos];
		if ( (domain_val & mask) != 0 )
		{
			this->neg_domain_augmentations.push(t_pos);
			this->neg_domain_augmentations.push(domain_val);

			/* remove this bit! */
			this->neg_domains[t_pos] = (char) (domain_val & (~mask));

			this->neg_domain_size_augmentations.push(t_pos);
			this->neg_domain_size_augmentations.push(domain_size);
			this->neg_domain_sizes[t_pos] = domain_size - 1;

			if ( domain_size - 1 == 1 )
			{
				/* The domain is a single element. Time to set that color! */
				for ( int i = 0; i < this->numColors; i++ )
				{
					if ( (this->neg_domains[t_pos] & (1 << i)) != 0 )
					{
						/* this color is in the domain! */
						bool result = this->setColor(pos, i, false);

						if ( !result )
						{
							return false;
						}

						return true;
					}
				}

				printf(
						"--[ProgressionTable::removeColorFromDomain] Should not have ended loop without finding element!\n");
				return false;
			}
			else if ( domain_size - 1 <= 0 )
			{
				/* out of colors! */
				return false;
			}
		}

		return true;
	}

	int domain_val = this->domains[pos];
	int domain_size = this->domain_sizes[pos];
	if ( (domain_val & mask) != 0 )
	{
		this->domain_augmentations.push(pos);
		this->domain_augmentations.push(domain_val);

		/* remove this bit! */
		this->domains[pos] = domain_val & (~mask);

		this->domain_size_augmentations.push(pos);
		this->domain_size_augmentations.push(domain_size);
		this->domain_sizes[pos] = domain_size - 1;

		if ( domain_size - 1 == 1 )
		{
			/* The domain is a single element. Time to set that color! */
			for ( int i = 0; i < this->numColors; i++ )
			{
				if ( (this->domains[pos] & (1 << i)) != 0 )
				{
					/* this color is in the domain! */
					bool result = this->setColor(pos, i, false);

					if ( !result )
					{
						return false;
					}

					return true;
				}
			}

			printf("--[ProgressionTable::removeColorFromDomain] Should not have ended loop without finding element!\n");
			return false;
		}
		else if ( domain_size - 1 <= 0 )
		{
			//			printf("--[ProgressionTable::removeColorFromDomain] out of colors!\n");
			/* no more colors? */
			return false;
		}
	}

	return true;
}

bool ProgressionTable::setDomainToSingleton(int pos, char color)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		printf("--[ProgressionTable::setDomainToSingleton(%d,%d)] Pos out of range.\n", pos, color);
		return false;
	}

	char mask = (1 << color);

	if ( pos < 0 )
	{
		if ( this->sym_mode == NO_SYMMETRY )
		{
			return false;
		}

		int t_pos = -1 - pos;

		int domain_val = this->neg_domains[t_pos];
		int domain_size = this->neg_domain_sizes[t_pos];
		if ( (domain_val & mask) != 0 )
		{
			this->neg_domain_augmentations.push(t_pos);
			this->neg_domain_augmentations.push(domain_val);

			/* replace with this bit! */
			this->neg_domains[t_pos] = mask;

			this->neg_domain_size_augmentations.push(t_pos);
			this->neg_domain_size_augmentations.push(domain_size);

			/* set to a single value */
			this->neg_domain_sizes[t_pos] = 1;

			/* we assume the color is set separately */
			return true;
		}
		else
		{
			/* can't set to singleton if it's not in the domain! */
			return false;
		}
	}

	int domain_val = this->domains[pos];
	int domain_size = this->domain_sizes[pos];
	if ( (domain_val & mask) != 0 )
	{
		this->domain_augmentations.push(pos);
		this->domain_augmentations.push(domain_val);

		/* replace with this bit! */
		this->domains[pos] = mask;

		this->domain_size_augmentations.push(pos);
		this->domain_size_augmentations.push(domain_size);

		/* set to have one element. */
		this->domain_sizes[pos] = 1;

		/* we assume the colors is set separately */
	}
	else
	{
		printf("--[ProgreesionTable::setDomainToSingleton] Color not in domain!\n");
		return false;
	}

	return true;
}

void ProgressionTable::snapshotDomains()
{
	this->domain_snapshots.push(this->domain_augmentations.size());
	this->neg_domain_snapshots.push(this->neg_domain_augmentations.size());
	this->domain_size_snapshots.push(this->domain_size_augmentations.size());
	this->neg_domain_size_snapshots.push(this->neg_domain_size_augmentations.size());
}

void ProgressionTable::rollbackDomains()
{
	int to_size = this->domain_snapshots.top();
	this->domain_snapshots.pop();

	while ( this->domain_augmentations.size() > to_size )
	{
		int to_val = this->domain_augmentations.top();
		this->domain_augmentations.pop();
		int to_pos = this->domain_augmentations.top();
		this->domain_augmentations.pop();

		this->domains[to_pos] = to_val;
	}

	to_size = this->neg_domain_snapshots.top();
	this->neg_domain_snapshots.pop();

	while ( this->neg_domain_augmentations.size() > to_size )
	{
		int to_val = this->neg_domain_augmentations.top();
		this->neg_domain_augmentations.pop();
		int to_pos = this->neg_domain_augmentations.top();
		this->neg_domain_augmentations.pop();

		this->neg_domains[to_pos] = to_val;
	}

	to_size = this->domain_size_snapshots.top();
	this->domain_size_snapshots.pop();

	while ( this->domain_size_augmentations.size() > to_size )
	{
		int to_val = this->domain_size_augmentations.top();
		this->domain_size_augmentations.pop();
		int to_pos = this->domain_size_augmentations.top();
		this->domain_size_augmentations.pop();

		this->domain_sizes[to_pos] = to_val;
	}

	to_size = this->neg_domain_size_snapshots.top();
	this->neg_domain_size_snapshots.pop();

	while ( this->neg_domain_size_augmentations.size() > to_size )
	{
		int to_val = this->neg_domain_size_augmentations.top();
		this->neg_domain_size_augmentations.pop();
		int to_pos = this->neg_domain_size_augmentations.top();
		this->neg_domain_size_augmentations.pop();

		this->neg_domain_sizes[to_pos] = to_val;
	}
}

/**
 * given the set of parameters (in Pseudo mode), return the size of the forward_table and backward_table arrays
 */
int ProgressionTable::getPseudoTableSize(int nColors)
{
	int size = nColors * this->maxN * this->maxL * (this->d + 1);

	return size;
}

/**
 * given the set of inputs (in Pseudo mode), return the index for the forward_table and backward_table arrays.
 */
int ProgressionTable::getPseudoTableIndex(int color, int pos, int l, int dp)
{
	int colorMult = this->maxN * this->maxL * (this->d + 1);
	int posMult = this->maxL * (this->d + 1);
	int lMult = (this->d + 1);

	if ( pos < 0 )
	{
		/* do a swap of coordinates */
		pos = -1 - pos;
	}

	int index = color * colorMult;
	index += pos * posMult;
	index += l * lMult;
	index += dp;

	return index;
}

/**
 * given the set of parameters (in Quasi mode), return the size of the forward_table and backward_table arrays
 */
int ProgressionTable::getQuasiTableSize(int nColors)
{
	int size = nColors * this->maxN * this->maxL;

	return size;
}

/**
 * given the set of inputs (in Quasi mode), return the index for the forward_table and backward_table arrays.
 */
int ProgressionTable::getQuasiTableIndex(int color, int pos, int l)
{
	int colorMult = this->maxN * this->maxL;
	int posMult = this->maxL;

	if ( pos < 0 )
	{
		/* do a swap of coordinates */
		pos = -1 - pos;
	}

	int index = color * colorMult;
	index += pos * posMult;
	index += l;

	return index;
}

void ProgressionTable::initTables()
{
	int table_size = this->getQuasiTableSize(this->numColors);

	if ( this->mode == PSEUDO_PROGRESSION )
	{
		table_size = this->getPseudoTableSize(this->numColors);
	}

	this->forward_table = (char*) malloc(table_size);
	this->neg_forward_table = (char*) malloc(table_size);
	this->backward_table = (char*) malloc(table_size);
	this->neg_backward_table = (char*) malloc(table_size);

	bzero(this->forward_table, table_size);
	bzero(this->neg_forward_table, table_size);
	bzero(this->backward_table, table_size);
	bzero(this->neg_backward_table, table_size);

	this->forward_table_snapshots.push(0);
	this->neg_forward_table_snapshots.push(0);
	this->backward_table_snapshots.push(0);
	this->neg_backward_table_snapshots.push(0);
}

void ProgressionTable::cleanTables()
{
	if ( this->forward_table != 0 )
	{
		free(this->forward_table);
		this->forward_table = 0;
	}

	if ( this->neg_forward_table != 0 )
	{
		free(this->neg_forward_table);
		this->neg_forward_table = 0;
	}

	if ( this->backward_table != 0 )
	{
		free(this->backward_table);
		this->backward_table = 0;
	}

	if ( this->neg_backward_table != 0 )
	{
		free(this->neg_backward_table);
		this->neg_backward_table = 0;
	}

	while ( this->forward_table_augmentations.size() > 0 )
	{
		this->forward_table_augmentations.pop();
	}

	while ( this->forward_table_snapshots.size() > 0 )
	{
		this->forward_table_snapshots.pop();
	}

	while ( this->neg_forward_table_augmentations.size() > 0 )
	{
		this->neg_forward_table_augmentations.pop();
	}

	while ( this->neg_forward_table_snapshots.size() > 0 )
	{
		this->neg_forward_table_snapshots.pop();
	}

	while ( this->backward_table_augmentations.size() > 0 )
	{
		this->backward_table_augmentations.pop();
	}

	while ( this->backward_table_snapshots.size() > 0 )
	{
		this->backward_table_snapshots.pop();
	}

	while ( this->neg_backward_table_augmentations.size() > 0 )
	{
		this->neg_backward_table_augmentations.pop();
	}

	while ( this->neg_backward_table_snapshots.size() > 0 )
	{
		this->neg_backward_table_snapshots.pop();
	}

}

void ProgressionTable::snapshotTables()
{
	this->forward_table_snapshots.push(this->forward_table_augmentations.size());
	this->neg_forward_table_snapshots.push(this->neg_forward_table_augmentations.size());
	this->backward_table_snapshots.push(this->backward_table_augmentations.size());
	this->neg_backward_table_snapshots.push(this->neg_backward_table_augmentations.size());
}

void ProgressionTable::rollbackTables()
{
	int to_size = this->forward_table_snapshots.top();
	this->forward_table_snapshots.pop();

	while ( this->forward_table_augmentations.size() > to_size )
	{
		int to_val = this->forward_table_augmentations.top();
		this->forward_table_augmentations.pop();
		int to_index = this->forward_table_augmentations.top();
		this->forward_table_augmentations.pop();

		this->forward_table[to_index] = to_val;
	}

	to_size = this->neg_forward_table_snapshots.top();
	this->neg_forward_table_snapshots.pop();

	while ( this->neg_forward_table_augmentations.size() > to_size )
	{
		int to_val = this->neg_forward_table_augmentations.top();
		this->neg_forward_table_augmentations.pop();
		int to_index = this->neg_forward_table_augmentations.top();
		this->neg_forward_table_augmentations.pop();

		this->neg_forward_table[to_index] = to_val;
	}

	to_size = this->backward_table_snapshots.top();
	this->backward_table_snapshots.pop();

	while ( this->backward_table_augmentations.size() > to_size )
	{
		int to_val = this->backward_table_augmentations.top();
		this->backward_table_augmentations.pop();
		int to_index = this->backward_table_augmentations.top();
		this->backward_table_augmentations.pop();

		this->backward_table[to_index] = to_val;
	}

	to_size = this->neg_backward_table_snapshots.top();
	this->neg_backward_table_snapshots.pop();

	while ( this->neg_backward_table_augmentations.size() > to_size )
	{
		int to_val = this->neg_backward_table_augmentations.top();
		this->neg_backward_table_augmentations.pop();
		int to_index = this->neg_backward_table_augmentations.top();
		this->neg_backward_table_augmentations.pop();

		this->neg_backward_table[to_index] = to_val;
	}
}

/**
 * setBackwardTable
 *
 * Set a position in the backward table, while updating the snapshot sizes. (Pseudo Mode)
 */
bool ProgressionTable::setBackwardTable(int color, int pos, int l, int dp, char value)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		return true;
	}

	/* can't look beyond the guaranteed spots */
	if ( pos <= -this->minSolN )
	{
		return true;
	}

	bool pos_in_kill_range = this->isInKillRange(pos);

	int index = this->getPseudoTableIndex(color, pos, l, dp);

	char* table = this->backward_table;

	if ( pos < 0 )
	{
		table = this->neg_backward_table;
	}

	if ( table[index] < value )
	{
		/* push index and previous value onto stack */
		if ( pos >= 0 )
		{
			this->backward_table_augmentations.push(index);
			this->backward_table_augmentations.push(table[index]);
		}
		else
		{
			this->neg_backward_table_augmentations.push(index);
			this->neg_backward_table_augmentations.push(table[index]);
		}

		/* now, set the value! */
		table[index] = value;
	}
	else
	{
		return true;
	}

	bool result = (value < this->k);

	if ( value == this->k - 1 )
	{
		if ( pos_in_kill_range && this->getColor(pos) == color )
		{
			this->has_progression = true;
			return false;
		}

		if ( this->prop_mode != NO_PROPAGATION )
		{
			/* we have value == this->k - 1 */
			/* SO, all NEXT steps can't be this color! */
			result &= this->removeColorFromDomain(pos, color);
		}
	}

	if ( this->getColor(pos) == color )
	{
		/* need to update NEXT positions */
		for ( int dd = 0; dd <= this->d - dp; dd++ )
		{
			result &= this->setBackwardTable(color, pos + (l + 1 + dd), l, dd + dp, value + 1);
		}
	}

	if ( result && this->prop_mode == FORWARD_BACKWARD_PROPAGATION )
	{
		int fval = this->getForwardTable(color, pos, l, this->d - dp);

		if ( value + fval >= this->k - 1 )
		{
			result &= this->removeColorFromDomain(pos, color);
		}
	}

	return !pos_in_kill_range || result;
}

/**
 * setBackwardTable
 *
 * Set a position in the backward table, while updating the snapshot sizes. (Quasi Mode)
 */
bool ProgressionTable::setBackwardTable(int color, int pos, int l, char value)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		return true;
	}

	/* can't look beyond the guaranteed spots */
	if ( pos <= -this->minSolN )
	{
		return true;
	}

	bool pos_in_kill_range = this->isInKillRange(pos);

	int index = this->getQuasiTableIndex(color, pos, l);

	char* table = this->backward_table;

	if ( pos < 0 )
	{
		table = this->neg_backward_table;
	}

	if ( table[index] < value )
	{
		//		for ( int i = 0; i < this->constraints.size(); i++ )
		//		{
		//			this->constraints[i]->addUpdatedBackwardTable(pos);
		//		}

		/* push index and previous value onto stack */
		if ( pos >= 0 )
		{
			this->backward_table_augmentations.push(index);
			this->backward_table_augmentations.push(table[index]);
		}
		else
		{
			this->neg_backward_table_augmentations.push(index);
			this->neg_backward_table_augmentations.push(table[index]);
		}

		/* now, set the value! */
		table[index] = value;
	}
	else
	{
		return true;
	}

	bool result = (value < this->k);

	if ( value == this->k - 1 )
	{
		if ( pos_in_kill_range && this->getColor(pos) == color )
		{
			this->has_progression = true;
			return false;
		}

		if ( this->prop_mode != NO_PROPAGATION )
		{
			/* we have value == this->k - 1 */
			/* SO, all NEXT steps can't be this color! */
			result = this->removeColorFromDomain(pos, color) && result;
		}
	}

	if ( this->getColor(pos) == color )
	{
		/* need to update NEXT position */
		for ( int dd = 0; result && dd <= this->d; dd++ )
		{
			result = this->setBackwardTable(color, pos + (l + 1 + dd), l, value + 1) && result;
		}
	}

	if ( result && this->prop_mode == FORWARD_BACKWARD_PROPAGATION )
	{
		int fval = this->getForwardTable(color, pos, l);

		if ( value + fval >= this->k - 1 )
		{
			result = this->removeColorFromDomain(pos, color) && result;
		}
	}

	return !pos_in_kill_range || result;
}

/* for all values in the current mode, push the backward table */
bool ProgressionTable::pushBackwardTable(int color, int pos)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		/* OUT OF RANGE */
		return true;
	}

	if ( this->getColor(pos) != color )
	{
		return true;
	}

	/* can't look beyond the guaranteed spots */
	if ( pos < -this->minSolN )
	{
		return true;
	}

	bool pos_in_kill_zone = this->isInKillRange(pos);

	/* okay, we have the same color, time to increase backward tables! */
	/* are we in range to push these values? */

	bool result = true;
	if ( this->mode == QUASI_PROGRESSION )
	{
		for ( int l = 0; result && l < this->maxL; l++ )
		{
			int val = this->getBackwardTable(color, pos, l);

			if ( val >= this->k - 1 )
			{
				/* too big! */
				if ( pos_in_kill_zone )
				{
					this->has_progression = true;
				}
				return false;
			}

			/* take a step back, add one, push value */
			for ( int D = 0; result && D <= this->d; D++ )
			{
				result = this->setBackwardTable(color, pos + (l + 1 + D), l, val + 1) && result;
			}
		}
	}
	else /* PSEUDO_PROGRESSION */
	{
		for ( int l = 0; result && l < this->maxL; l++ )
		{
			for ( int dp = 0; result && dp <= this->d; dp++ )
			{
				int val = this->getBackwardTable(color, pos, l, dp);

				if ( val >= this->k - 1 )
				{
					/* too big! */
					if ( pos_in_kill_zone )
					{
						this->has_progression = true;
					}

					return false;
				}

				/* take a step back, add one, push value */
				for ( int D = 0; result && D <= this->d - dp; D++ )
				{
					result = this->setBackwardTable(color, pos + (l + 1 + D), l, dp + D, val + 1) && result;
				}
			}
		}
	}

	if ( pos_in_kill_zone && !result )
	{
		return false;
	}

	return true;
}

/**
 * getBackwardTable
 *
 * Returns the value of the backward table.
 */
char ProgressionTable::getBackwardTable(int color, int pos, int l, int dp)
{
	int index = this->getPseudoTableIndex(color, pos, l, dp);

	if ( pos >= 0 )
	{
		return this->backward_table[index];
	}
	else
	{
		return this->neg_backward_table[index];
	}
}
/**
 * getBackwardTable
 *
 * Returns the value of the backward table.
 */
char ProgressionTable::getBackwardTable(int color, int pos, int l)
{
	int index = this->getQuasiTableIndex(color, pos, l);

	if ( pos >= 0 )
	{
		return this->backward_table[index];
	}
	else
	{
		return this->neg_backward_table[index];
	}
}

/**
 * setForwardTable
 *
 * Set a position in the forward table, while updating the snapshot sizes. (Pseudo Mode)
 */
bool ProgressionTable::setForwardTable(int color, int pos, int l, int dp, char value)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		return true;
	}

	/* can't look beyond the guaranteed spots */
	if ( pos >= this->minSolN )
	{
		return true;
	}

	if ( this->sym_mode == NO_SYMMETRY && pos < this->curN - 1 )
	{
		/* don't bother setting values here! */
		return true;
	}

	bool pos_in_kill_range = this->isInKillRange(pos);

	int index = this->getPseudoTableIndex(color, pos, l, dp);

	char* table = this->forward_table;

	if ( pos < 0 )
	{
		table = this->neg_forward_table;
	}

	if ( table[index] > value )
	{
		/* no updates... */
		return true;
	}

	if ( table[index] < value )
	{
		//		for ( int i = 0; i < this->constraints.size(); i++ )
		//		{
		//			this->constraints[i]->addUpdatedForwardTable(pos);
		//		}

		/* push index and previous value onto stack */
		if ( pos >= 0 )
		{
			this->forward_table_augmentations.push(index);
			this->forward_table_augmentations.push(table[index]);
		}
		else
		{
			this->neg_forward_table_augmentations.push(index);
			this->neg_forward_table_augmentations.push(table[index]);
		}

		/* now, set the value! */
		table[index] = value;
	}
	else
	{
		return true;
	}

	bool result = (value < this->k);

	if ( value == this->k - 1 )
	{
		if ( pos_in_kill_range && this->getColor(pos) == color )
		{
			this->has_progression = true;
			return false;
		}

		/* this color would finish the job */
		result &= this->removeColorFromDomain(pos, color);
	}

	if ( this->getColor(pos) == color )
	{
		/* push to NEXT positions */
		for ( int dd = 0; result && dd <= this->d - dp; dd++ )
		{
			result &= this->setForwardTable(color, pos - (l + 1 + dd), l, dp + dd, value + 1);
		}
	}

	if ( this->prop_mode == FORWARD_BACKWARD_PROPAGATION )
	{
		int bval = this->getBackwardTable(color, pos, l, this->d - dp);

		if ( value + bval >= this->k - 1 )
		{
			result &= this->removeColorFromDomain(pos, color);
		}
	}

	return !pos_in_kill_range || result;
}

/**
 * setForwardTable
 *
 * Set a position in the forward table, while updating the snapshot sizes. (Quasi Mode)
 */
bool ProgressionTable::setForwardTable(int color, int pos, int l, char value)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		//printf("--[ProgressionTable::setForwardTable(%d,%d,%d,%d)] Out of bounds!\n", color, pos, l, value);
		return true;
	}

	/* can't look beyond the guaranteed spots */
	if ( pos >= this->minSolN )
	{
		return true;
	}

	if ( this->sym_mode == NO_SYMMETRY && pos < this->curN - 1 )
	{
		/* don't bother setting values here! */
		return true;
	}

	bool pos_in_kill_range = this->isInKillRange(pos);
	int index = this->getQuasiTableIndex(color, pos, l);

	char* table = this->forward_table;

	if ( pos < 0 )
	{
		table = this->neg_forward_table;
	}

	if ( table[index] < value )
	{
		for ( int i = 0; i < this->constraints.size(); i++ )
		{
			this->constraints[i]->addUpdatedForwardTable(pos);
		}

		/* push index and previous value onto stack */
		if ( pos >= 0 )
		{
			this->forward_table_augmentations.push(index);
			this->forward_table_augmentations.push(table[index]);
		}
		else
		{
			this->neg_forward_table_augmentations.push(index);
			this->neg_forward_table_augmentations.push(table[index]);
		}

		/* now, set the value! */
		table[index] = value;
	}
	else
	{
		return true;
	}

	bool result = (value < this->k);

	if ( value == this->k - 1 )
	{
		if ( pos_in_kill_range && this->getColor(pos) == color )
		{
			this->has_progression = true;
			return false;
		}

		/* this color would finish the job */
		result &= this->removeColorFromDomain(pos, color);
	}

	if ( this->getColor(pos) == color )
	{
		/* push to NEXT positions */
		for ( int dd = 0; result && dd <= this->d; dd++ )
		{
			result &= this->setForwardTable(color, pos - (l + 1 + dd), l, value + 1);
		}
	}

	if ( this->prop_mode == FORWARD_BACKWARD_PROPAGATION )
	{
		int bval = this->getBackwardTable(color, pos, l);

		if ( value + bval >= this->k - 1 )
		{
			result &= this->removeColorFromDomain(pos, color);
		}
	}

	return !pos_in_kill_range || result;
}

/* for all values in the current mode, push the forward table */
bool ProgressionTable::pushForwardTable(int color, int pos)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		/* OUT OF RANGE */
		return true;
	}

	if ( this->getColor(pos) != color )
	{
		return true;
	}

	/* can't look beyond the guaranteed spots */
	if ( pos >= this->minSolN )
	{
		return true;
	}

	/* okay, we have the same color, time to increase forward tables! */
	/* are we in range to push these values? */

	bool result = true;
	if ( this->mode == QUASI_PROGRESSION )
	{
		for ( int l = 0; l < this->maxL; l++ )
		{
			int val = this->getForwardTable(color, pos, l);

			/* take a step back, add one, push value */
			for ( int D = 0; D <= this->d; D++ )
			{
				result = this->setForwardTable(color, pos - (l + 1 + D), l, val + 1) && result;
			}
		}
	}
	else /* PSEUDO_PROGRESSION */
	{
		for ( int l = 0; l < this->maxL; l++ )
		{
			for ( int dp = 0; dp <= this->d; dp++ )
			{
				int val = this->getForwardTable(color, pos, l, dp);

				/* take a step back, add one, push value */
				for ( int D = 0; D <= this->d - dp; D++ )
				{
					result = this->setForwardTable(color, pos - (l + 1 + D), l, dp + D, val + 1) && result;
				}
			}
		}
	}

	if ( !result && this->isInKillRange(pos) )
	{
		return false;
	}

	return true;
}

/**
 * getForwardTable
 *
 * Returns the value of the forward table. (Pseudo Mode)
 */
char ProgressionTable::getForwardTable(int color, int pos, int l, int dp)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		return -1;
	}

	int index = this->getPseudoTableIndex(color, pos, l, dp);

	if ( pos >= 0 )
	{
		return this->forward_table[index];
	}
	else
	{
		return this->neg_forward_table[index];
	}
}

/**
 * getForwardTable
 *
 * Returns the value of the forward table. (Quasi Mode)
 */
char ProgressionTable::getForwardTable(int color, int pos, int l)
{
	if ( pos < this->getMinPos() || pos >= this->getMaxPos() )
	{
		return -1;
	}

	int index = this->getQuasiTableIndex(color, pos, l);

	if ( pos >= 0 )
	{
		return this->forward_table[index];
	}
	else
	{
		return this->neg_forward_table[index];
	}
}

void ProgressionTable::init()
{
	this->initColors();
	this->initDomains();
	this->initTables();

	this->setColor(0, 0, true);

	this->snapshot();

	this->propagate();

	this->snapshot();
}

void ProgressionTable::clean()
{
	this->cleanColors();
	this->cleanDomains();
	this->cleanTables();
}

bool ProgressionTable::isInKillRange(int pos)
{
	return ((pos < this->minSolN) && (pos > -1 - this->minSolN));
}

/**
 * ProgressionTable constructor
 */
ProgressionTable::ProgressionTable(progression_mode mode, int k, int d, int minN, int maxN, int r,
		symmetry_mode sym_mode, propagation_mode prop_mode)
{
	this->mode = mode;
	this->sym_mode = sym_mode;
	this->prop_mode = prop_mode;
	this->k = k;
	this->d = d;
	this->minN = minN;
	this->minSolN = minN;
	this->maxN = maxN;
	this->numColors = r;
	this->curN = 0;

	this->use_backward_propagation = false;
	this->use_forward_propagation = false;

	this->maxL = maxN / (k - 1);

	if ( sym_mode != NO_SYMMETRY )
	{
		this->maxL = (2 * maxN) / (k - 1);
	}

	this->has_progression = false;

	this->init();
}

/**
 * Destructor
 */
ProgressionTable::~ProgressionTable()
{
	this->clean();
}

/**
 * getN -- get the current level of colors.
 */
int ProgressionTable::getN()
{
	return this->curN;
}

/**
 * getColor
 */
char ProgressionTable::getColor(int j)
{
	if ( j < this->getMinPos() || j >= this->getMaxPos() )
	{
		return UNKNOWN;
	}

	if ( j >= 0 )
	{
		return this->colors[j];
	}
	else if ( this->sym_mode == SYMMETRIC )
	{
		return this->colors[-1 - j];
	}
	else if ( this->sym_mode == SKEW_SYMMETRIC )
	{
		/* swap colors! */
		if ( this->colors[-1 - j] != (char) UNKNOWN )
		{
			return 1 - this->colors[-1 - j];
		}
		else
		{
			return UNKNOWN;
		}
	}

	return UNKNOWN;
}

/**
 * setColor
 *
 * @return false if the color is incorrect or if the propagation fails.
 */
bool ProgressionTable::setColor(int j, char c, bool top_level)
{
//	if ( top_level )
//	{
//		printf("--[ProgressionTable::setColor(%d,%d,...)] called \n", j, c);
//	}

	if ( j < this->getMinPos() || j >= this->getMaxPos() )
	{
		printf("--[ProgressionTable::setColor(%d,%d,...)] position out of range! (Should be within [%d, %d) \n", j, c,
				this->getMinPos(), this->getMaxPos());
		return false;
	}

	if ( c < 0 || c >= this->numColors )
	{
		printf("--[ProgressionTable::setColor(%d,%d,...)] color out of range!\n", j, c);
		return false;
	}

	bool pos_in_kill_range = top_level || this->isInKillRange(j);

	int c_index1 = j;
	int c_index2 = j;

	int c1 = c;
	int c2 = c;

	if ( j < 0 )
	{
		if ( this->sym_mode == NO_SYMMETRY )
		{
			printf("--[ProgressionTable::setColor(%d, %d, ...)] j is negative but we have no symmetry!\n", j, c);
			return false;
		}
		else
		{
			c_index1 = -1 - j;

			if ( this->sym_mode == SKEW_SYMMETRIC )
			{
				/* toggle colors (two-color mode) */
				c1 = 1 - c;
			}
		}
	}
	else
	{
		if ( this->sym_mode == NO_SYMMETRY )
		{
			/* no problem here! */
		}
		else
		{
			c_index2 = -1 - j;

			if ( this->sym_mode == SKEW_SYMMETRIC )
			{
				/* toggle colors (two-color mode) */
				c2 = 1 - c;
			}
		}
	}

	/* check color! */
	if ( this->colors[c_index1] == (char) UNKNOWN )
	{
		char dom1 = this->getDomain(c_index1);

		bool result = true;

		if ( pos_in_kill_range && (dom1 & (1 << c1)) == 0 )
		{
			/* we don't have this color in our domain! */
//			printf("--[ProgressionTable::setColor(%d, %d, ...)] Color %d is not in the 1st domain: %X!\n", j, c, c1,
//					dom1);
			return false;
		}

		if ( this->sym_mode != NO_SYMMETRY )
		{
			char dom2 = this->getDomain(c_index2);

			if ( pos_in_kill_range && (dom2 & (1 << c2)) == 0 )
			{
				/* we don't have this color in our domain! */
//				printf("--[ProgressionTable::setColor(%d, %d, ...)] Color %d is not in the 2nd domain: %X!\n", j, c, c2,
//						dom2);
				return false;
			}

			/* set color! */
			result &= this->setDomainToSingleton(c_index2, c2);
		}

		/* set color! */
		this->colors[c_index1] = c1;
		this->color_augmentations.push(c_index1);
		result &= this->setDomainToSingleton(c_index1, c1);

		if ( top_level )
		{
			this->curN = c_index1 + 1;
		}

		if ( result && this->sym_mode != NO_SYMMETRY )
		{
			result &= this->pushBackwardTable(c2, c_index2);

//			if ( !result )
//			{
//				printf("--push backward c2 %d %d.\n", c2, c_index2);
//			}
		}

		result &= this->pushBackwardTable(c1, c_index1);

//		if ( !result )
//		{
//			printf("--push backward c1 %d %d.\n", c1, c_index1);
//			this->print();
//		}

		if ( result && this->sym_mode != NO_SYMMETRY || this->prop_mode == FORWARD_BACKWARD_PROPAGATION )
		{
			result &= this->pushForwardTable(c1, c_index1);
//			if ( !result )
//			{
//				printf("--push forward c1 %d %d.\n", c1, c_index1);
//			}
		}

		if ( result && this->sym_mode != NO_SYMMETRY )
		{
			result &= this->pushForwardTable(c2, c_index2);

//			if ( !result )
//			{
//				printf("--push forward c2 %d %d.\n", c2, c_index2);
//			}
		}

		if ( pos_in_kill_range && !result )
		{
			this->has_progression = true;
			return false;
		}

		return true;
	}
	else if ( this->colors[c_index1] != c1 )
	{
		return !pos_in_kill_range || false;
	}
	else
	{
		if ( top_level )
		{
			this->curN = c_index1 + 1;
		}
		/* color already set! */
	}

	return true;
}

/**
 * hasProgression
 */
bool ProgressionTable::hasProgression()
{
	if ( this->has_progression )
	{
//		printf("-- IMMEDIATE PROGRESSION.\n");
		return true;
	}

	if ( this->minSolN < this->curN && this->mode == QUASI_PROGRESSION )
	{
		/* only check the positions which previously didn't send a kill! */
		for ( int i = this->minSolN - 1; i < this->curN; i++ )
		{
			if ( i < 0 )
			{
				continue;
			}

			for ( int c = 0; c < this->numColors; c++ )
			{
				for ( int l = 0; l < this->maxL; l++ )
				{
					int val = this->getBackwardTable(c, i, l);
					if ( this->colors[i] == c && val >= k - 1 )
					{
//						printf("-- COLOR AND VALUE pos = %d.\n", i);
						this->has_progression = true;
						return true;
					}

					if ( val >= k )
					{
//						printf("-- JUST VALUE pos = %d.\n", i);
						this->has_progression = true;
						return true;
					}
				}
			}

			this->minSolN = i + 1;

			/* these were not pushed earlier, so push them now! */
			bool result = this->pushBackwardTable(this->getColor(i), i);

			if ( result && this->sym_mode != NO_SYMMETRY )
			{
				result = this->pushBackwardTable(this->getColor(-1 - i), -1 - i);
			}

			if ( !result )
			{
				return true;
			}
		}

		/* update! this is a good solution! */
		this->minSolN = this->curN;

	} /* end QUASI case */
	else if ( this->minSolN < this->curN && this->mode == PSEUDO_PROGRESSION )
	{
		/* only check the positions which previously didn't send a kill! */
		for ( int i = this->minSolN - 1; i < this->curN; i++ )
		{
			if ( i < 0 )
			{
				continue;
			}

			for ( int c = 0; c < this->numColors; c++ )
			{
				for ( int l = 0; l < this->maxL; l++ )
				{
					for ( int D = 0; D <= this->d; D++ )
					{
						int val = this->getBackwardTable(c, i, l, D);
						if ( val >= k - 1 && this->colors[i] == c )
						{
							this->has_progression = true;
							return true;
						}

						if ( val >= k )
						{
							this->has_progression = true;
							return true;
						}
					}
				}
			}

			this->minSolN = i + 1;

			/* these were not pushed earlier, so push them now! */
			bool result = this->pushBackwardTable(this->getColor(i), i);

			if ( result && this->sym_mode != NO_SYMMETRY )
			{
				result = this->pushBackwardTable(this->getColor(-1 - i), -1 - i);
			}

			if ( !result )
			{
				return true;
			}
		}
		/* update! this is a good solution! */
		this->minSolN = this->curN;
	} /* end PSEUDO case */

	return false;
}

/**
 * snapshot
 */
/**
 * rollback
 */
void ProgressionTable::snapshot()
{
	this->curn_snapshots.push(this->curN);
	this->snapshotColors();
	this->snapshotDomains();
	this->snapshotTables();
}

void ProgressionTable::rollback()
{
	this->has_progression = false;

	this->rollbackTables();
	this->rollbackDomains();
	this->rollbackColors();

	this->curN = this->curn_snapshots.top();

	this->curn_snapshots.pop();

	/* clear out things to propagate */
//	for ( int i = 0; i < this->constraints.size(); i++ )
//	{
//		this->constraints[i]->empty();
//	}
}

/**
 * setBackwardPropagation()
 */
void ProgressionTable::setBackwardPropagation(bool val)
{
	this->use_backward_propagation = val;

	if ( val == true )
	{
		if ( this->mode == PSEUDO_PROGRESSION )
		{
			//			this->constraints.push_back(new PseudoBackwardConstraint(this));
			//			this->constraints.push_back(new PseudoBDomainConstraint(this));
			//			this->constraints.push_back(new PseudoFBDomainConstraint(this));
			//			this->constraints.push_back(new PseudoForwardConstraint(this));
		}
		else if ( this->mode == QUASI_PROGRESSION )
		{
			//			this->constraints.push_back(new QuasiBackwardConstraint(this));
			//			this->constraints.push_back(new QuasiBDomainConstraint(this));
			//			this->constraints.push_back(new QuasiFBDomainConstraint(this));
			//			this->constraints.push_back(new QuasiForwardConstraint(this));
		}
	}
}

/**
 * setForwardPropagation()
 */
void ProgressionTable::setForwardPropagation(bool val)
{
	this->use_forward_propagation = val;

	if ( val == true )
	{
		if ( this->mode == PSEUDO_PROGRESSION )
		{
			//			this->constraints.push_back(new PseudoBackwardConstraint(this));
			//			this->constraints.push_back(new PseudoBDomainConstraint(this));
			//			this->constraints.push_back(new PseudoForwardConstraint(this));
			//			this->constraints.push_back(new PseudoFBDomainConstraint(this));
		}
		else if ( this->mode == QUASI_PROGRESSION )
		{
			//			this->constraints.push_back(new QuasiBackwardConstraint(this));
			//			this->constraints.push_back(new QuasiBDomainConstraint(this));
			//			this->constraints.push_back(new QuasiForwardConstraint(this));
			//			this->constraints.push_back(new QuasiFBDomainConstraint(this));
		}
	}
}

/**
 * propagate : run through the constraints until
 *
 * a) stable
 * b) contradiction.
 */
bool ProgressionTable::propagate()
{
	bool result = true;

	if ( this->has_progression )
	{
		return false;
	}

//	bool some_constraint_nonempty = true;
//
//	while ( result && some_constraint_nonempty )
//	{
//		some_constraint_nonempty = false;
//
//		for ( int i = 0; result && i < this->constraints.size(); i++ )
//		{
//			if ( this->constraints[i]->isEmpty() == false )
//			{
//				some_constraint_nonempty = true;
//				bool temp_result = this->constraints[i]->doAllPropagation();
//
//				if ( !temp_result )
//				{
////					printf("-- propagation constraint %d failed.\n", i);
//				}
//				result = temp_result && result;
//			}
//		}
//	}
//
//	if ( !result )
//	{
//		this->has_progression = true;
//	}

	return result;
}

/**
 * print
 */
void ProgressionTable::print()
{
	printf("--[ProgressionTable::print()]\t");

	if ( this->has_progression )
	{
		printf("has progression.");
	}

	printf("\n");
	for ( int i = this->getMinPos(); i < this->getMaxPos(); i++ )
	{
		if ( this->sym_mode != NO_SYMMETRY && i == 0 )
		{
			printf("<>");
		}
		else if ( i == this->minSolN || i == -this->minSolN )
		{
			printf("|");
		}
		else if ( i == this->curN || i == -this->curN )
		{
			printf(":");
		}
		else
		{
			printf(" ");
		}

		printf("%d", this->getColor(i));
	}
	printf("\n");

	printf("domains:\n");

	for ( int i = this->getMinPos(); i < this->getMaxPos(); i++ )
	{
		if ( this->sym_mode != NO_SYMMETRY && i == 0 )
		{
			printf("<>");
		}
		else if ( i == this->minSolN || i == -this->minSolN )
		{
			printf("|");
		}
		else if ( i == this->curN || i == -this->curN )
		{
			printf(":");
		}
		else
		{
			printf(" ");
		}

		printf("%d", this->getDomain(i));
	}
	printf("\n");

	if ( this->use_forward_propagation )
	{
		printf("forward (max per color):\n");

		for ( char c = 0; c < this->numColors; c++ )
		{
			if ( this->sym_mode != NO_SYMMETRY )
			{
				for ( int i = this->maxN - 1; i >= 0; i-- )
				{
					if ( i == this->minSolN - 1 )
					{
						printf("|");
					}
					else if ( i == this->curN - 1 )
					{
						printf(":");
					}
					else
					{
						printf(" ");
					}

					int max = 0;
					if ( this->mode == QUASI_PROGRESSION )
					{
						for ( int l = 0; l < this->maxL; l++ )
						{
							int val = this->getForwardTable(c, -1 - i, l);

							if ( val > max )
								max = val;
						}
					}
					else
					{
						for ( int l = 0; l < this->maxL; l++ )
						{
							for ( int dp = 0; dp <= this->d; dp++ )
							{
								int val = this->getForwardTable(c, -1 - i, l, dp);

								if ( val > max )
									max = val;
							}
						}
					}
					printf("%d", max);
				}
				printf("<>");
			}
			for ( int i = 0; i < this->maxN; i++ )
			{
				if ( i == this->minSolN )
				{
					printf("|");
				}
				else if ( i == this->curN )
				{
					printf(":");
				}
				else if ( i > 0 )
				{
					printf(" ");
				}
				int max = 0;
				if ( this->mode == QUASI_PROGRESSION )
				{
					for ( int l = 0; l < this->maxL; l++ )
					{
						int val = this->getForwardTable(c, i, l);

						if ( val > max )
							max = val;
					}
				}
				else
				{
					for ( int l = 0; l < this->maxL; l++ )
					{
						for ( int dp = 0; dp <= this->d; dp++ )
						{
							int val = this->getForwardTable(c, i, l, dp);

							if ( val > max )
								max = val;
						}
					}
				}
				printf("%d", max);
			}
			printf("\n");
		}
	}

	printf("backward (max by color):\n");
	for ( char c = 0; c < this->numColors; c++ )
	{
		if ( this->sym_mode != NO_SYMMETRY )
		{
			for ( int i = this->maxN - 1; i >= 0; i-- )
			{
				if ( i == this->minSolN - 1 )
				{
					printf("|");
				}
				else if ( i == this->curN - 1 )
				{
					printf(":");
				}
				else
				{
					printf(" ");
				}

				int max = 0;
				if ( this->mode == QUASI_PROGRESSION )
				{
					for ( int l = 0; l < this->maxL; l++ )
					{
						int val = this->getBackwardTable(c, -1 - i, l);

						if ( val > max )
							max = val;
					}
				}
				else
				{
					for ( int l = 0; l < this->maxL; l++ )
					{
						for ( int dp = 0; dp <= this->d; dp++ )
						{
							int val = this->getBackwardTable(c, -1 - i, l, dp);

							if ( val > max )
								max = val;
						}
					}
				}
				printf("%d", max);
			}
			printf("<>");
		}
		for ( int i = 0; i < this->maxN; i++ )
		{
			if ( i == this->minSolN )
			{
				printf("|");
			}
			else if ( i == this->curN || i == -1 - this->curN )
			{
				printf(":");
			}
			else if ( i > 0 )
			{
				printf(" ");
			}

			int max = 0;
			if ( this->mode == QUASI_PROGRESSION )
			{
				for ( int l = 0; l < this->maxL; l++ )
				{
					int val = this->getBackwardTable(c, i, l);

					if ( val > max )
						max = val;
				}
			}
			else
			{
				for ( int l = 0; l < this->maxL; l++ )
				{
					for ( int dp = 0; dp <= this->d; dp++ )
					{
						int val = this->getBackwardTable(c, i, l, dp);

						if ( val > max )
							max = val;
					}
				}
			}
			printf("%d", max);
		}
		printf("\n");
	}

	printf("\n");
	fflush(stdout);
}

void ProgressionTable::printSolution()
{
	int min_i = 0;

	if ( this->sym_mode != NO_SYMMETRY )
	{
		min_i = -this->getN();
	}

	for ( int i = min_i; i < this->getN(); i++ )
	{
		int c = this->getColor(i);
		int num_block = 1;

		while ( i + 1 < this->getN() && this->getColor(i + 1) == c )
		{
			i++;
			num_block++;
		}

		printf(" %d^{%d} ", c, num_block);
	}
	printf("\n\n");
}

void ProgressionTable::reset()
{
	this->clean();
	this->init();
}

int ProgressionTable::getNumColors()
{
	return this->numColors;
}

int ProgressionTable::getMinSolN()
{
	return this->minSolN;
}

int ProgressionTable::getMaxL()
{
	return this->maxL;
}

int ProgressionTable::getK()
{
	return this->k;
}

int ProgressionTable::getD()
{
	return this->d;
}

