/*
 * ProgressionTable.hpp
 *
 *  Created on: May 15, 2011
 *      Author: derrickstolee
 */

#ifndef PROGRESSIONTABLE_HPP_
#define PROGRESSIONTABLE_HPP_

#include <stack>
#include <vector>
#include "ProgressionConstraint.hpp"

typedef enum symmetry_enum
{
	NO_SYMMETRY, SYMMETRIC, SKEW_SYMMETRIC
} symmetry_mode;

typedef enum progression_enum
{
	QUASI_PROGRESSION, PSEUDO_PROGRESSION
} progression_mode;

typedef enum propagation_enum
{
	NO_PROPAGATION, BACKWARD_PROPAGATION, FORWARD_BACKWARD_PROPAGATION
} propagation_mode;

#define UNKNOWN 0xF

class ProgressionTable
{
protected:
	progression_mode mode;

	int k;
	int d;
	int minN;
	int minSolN;
	int maxN;
	int curN;
	int numColors;

	bool use_backward_propagation;
	bool use_forward_propagation;
	symmetry_mode sym_mode;
	propagation_mode prop_mode;

	int maxL;

	bool has_progression;

	std::vector<ProgressionConstraint*> constraints;

	/**
	 * Colors stores the assigned colors.
	 */
	char* colors;
	std::stack<int> color_augmentations;
	std::stack<int> color_snapshots;
	std::stack<int> curn_snapshots;
	void initColors();
	void cleanColors();
	void snapshotColors();
	void rollbackColors();

	/**
	 * A domain is a bit-mask for possible colors to go.
	 */
	char* domains;
	char* neg_domains;
	char* domain_sizes;
	char* neg_domain_sizes;
	std::stack<int> domain_augmentations; /* positions and previous values come in pairs [2*i], [2*i+1]. */
	std::stack<int> domain_snapshots;
	std::stack<int> neg_domain_augmentations; /* positions and previous values come in pairs [2*i], [2*i+1]. */
	std::stack<int> neg_domain_snapshots;
	std::stack<int> domain_size_augmentations; /* positions and previous values come in pairs [2*i], [2*i+1]. */
	std::stack<int> domain_size_snapshots;
	std::stack<int> neg_domain_size_augmentations; /* positions and previous values come in pairs [2*i], [2*i+1]. */
	std::stack<int> neg_domain_size_snapshots;
	void initDomains();
	void cleanDomains();
	char getDomain(int pos);
	bool setDomainToSingleton(int pos, char color); /* this is called by setColor */
	void snapshotDomains();
	void rollbackDomains();

	char* forward_table;
	char* neg_forward_table;
	char* backward_table;
	char* neg_backward_table;
	std::stack<int> forward_table_augmentations; /* indices and previous values */
	std::stack<int> forward_table_snapshots; /* stack of sizes */
	std::stack<int> neg_forward_table_augmentations; /* indices and previous values */
	std::stack<int> neg_forward_table_snapshots; /* stack of sizes */
	std::stack<int> backward_table_augmentations; /* indices and previous values */
	std::stack<int> backward_table_snapshots; /* stack of sizes */
	std::stack<int> neg_backward_table_augmentations; /* indices and previous values */
	std::stack<int> neg_backward_table_snapshots; /* stack of sizes */

	void initTables();
	void cleanTables();
	void snapshotTables();
	void rollbackTables();

	/**
	 * given the set of parameters (in Pseudo mode), return the size of the forward_table and backward_table arrays
	 */
	int getPseudoTableSize(int nColors);

	/**
	 * given the set of inputs (in Pseudo mode), return the index for the forward_table and backward_table arrays.
	 */
	int getPseudoTableIndex(int color, int pos, int l, int dp);

	/**
	 * given the set of parameters (in Quasi mode), return the size of the forward_table and backward_table arrays
	 */
	int getQuasiTableSize(int nColors);

	/**
	 * given the set of inputs (in Quasi mode), return the index for the forward_table and backward_table arrays.
	 */
	int getQuasiTableIndex(int color, int pos, int l);

	/* for all values in the current mode, push the backward table */
	bool pushBackwardTable(int color, int pos);

	/* for all values in the current mode, push the backward table */
	bool pushForwardTable(int color, int pos);

	void init();
	void clean();

public:
	/**
	 * ProgressionTable constructor
	 */
	ProgressionTable(progression_mode mode, int k, int d, int minN, int maxN, int r, symmetry_mode sym_mode,
			propagation_mode prop_mode);

	/**
	 * Destructor
	 */
	virtual ~ProgressionTable();

	/**
	 * getN -- get the current level of colors.
	 */
	int getN();

	/**
	 * getColor
	 */
	char getColor(int j);

	/**
	 * setColor
	 *
	 * @return false if the color is incorrect or if the propagation fails.
	 */
	bool setColor(int j, char c, bool top_level);

	/**
	 * hasProgression
	 */
	bool hasProgression();

	/**
	 * snapshot
	 */
	void snapshot();

	/**
	 * rollback
	 */
	void rollback();

	/**
	 * setBackwardPropagation()
	 */
	void setBackwardPropagation(bool val);

	/**
	 * setForwardPropagation()
	 */
	void setForwardPropagation(bool val);

	/**
	 * propagate : run through the constraints until
	 *
	 * a) stable
	 * b) contradiction.
	 */
	bool propagate();

	/**
	 * print
	 */
	void print();

	void printSolution();

	void reset();
	int getMinPos();
	int getMaxPos();

	/**
	 * setBackwardTable
	 *
	 * Set a position in the backward table, while updating the snapshot sizes. (Pseudo Mode)
	 */
	bool setBackwardTable(int color, int pos, int l, int dp, char value);

	/**
	 * setBackwardTable
	 *
	 * Set a position in the backward table, while updating the snapshot sizes. (Quasi Mode)
	 */
	bool setBackwardTable(int color, int pos, int l, char value);

	/**
	 * getBackwardTable
	 *
	 * Returns the value of the backward table. (Pseudo Mode)
	 */
	char getBackwardTable(int color, int pos, int l, int dp);

	/**
	 * getBackwardTable
	 *
	 * Returns the value of the backward table. (Quasi Mode)
	 */
	char getBackwardTable(int color, int pos, int l);

	/**
	 * setForwardTable
	 *
	 * Set a position in the forward table, while updating the snapshot sizes. (Pseudo Mode)
	 */
	bool setForwardTable(int color, int pos, int l, int dp, char value);

	/*
	 * setForwardTable
	 *
	 * Set a position in the forward table, while updating the snapshot sizes. (Quasi Mode)
	 */
	bool setForwardTable(int color, int pos, int l, char value);

	/**
	 * getForwardTable
	 *
	 * Returns the value of the forward table. (Pseudo Mode)
	 */
	char getForwardTable(int color, int pos, int l, int dp);

	/**
	 * getForwardTable
	 *
	 * Returns the value of the forward table. (Quasi Mode)
	 */
	char getForwardTable(int color, int pos, int l);

	int getNumColors();
	int getMinSolN();
	int getMaxL();
	int getK();
	int getD();
	bool isInKillRange(int pos);
	bool removeColorFromDomain(int pos, char color); /* this function may call setColor */

};

#endif /* PROGRESSIONTABLE_HPP_ */
